
import React, { useState, useCallback } from 'react';
import type { RouteInput, RouteResult } from './types';
import { getRouteInformation } from './services/geminiService';
import Header from './components/Header';
import RouteInputTable from './components/RouteInputTable';
import ResultsTable from './components/ResultsTable';
import Loader from './components/Loader';
import { PlusIcon } from './components/icons/PlusIcon';
import FileUploadButton from './components/FileUploadButton';

const App: React.FC = () => {
  const [routes, setRoutes] = useState<RouteInput[]>([
    { id: 1, from: 'Delhi', to: 'Mumbai', actualDistance: '1420' },
    { id: 2, from: 'Bengaluru', to: 'Hyderabad', actualDistance: '575' },
  ]);
  const [results, setResults] = useState<RouteResult[]>([]);
  const [isLoading, setIsLoading] = useState<boolean>(false);
  const [error, setError] = useState<string | null>(null);
  const [fileError, setFileError] = useState<string | null>(null);

  const handleAddRoute = useCallback(() => {
    setRoutes(prev => [...prev, { id: Date.now(), from: '', to: '', actualDistance: '' }]);
  }, []);

  const handleRouteChange = useCallback((id: number, field: keyof Omit<RouteInput, 'id'>, value: string) => {
    setRoutes(prev => prev.map(route => route.id === id ? { ...route, [field]: value } : route));
  }, []);

  const handleRemoveRoute = useCallback((id: number) => {
    setRoutes(prev => prev.filter(route => route.id !== id));
  }, []);

  const handleFileSelect = useCallback((file: File) => {
    setFileError(null);
    if (!file) return;

    if (!file.type.includes('csv') && !file.name.endsWith('.csv')) {
        setFileError('Invalid file type. Please upload a CSV file.');
        return;
    }

    const reader = new FileReader();
    reader.onload = (e) => {
        const text = e.target?.result as string;
        if (!text) {
            setFileError('File is empty or could not be read.');
            return;
        }
        try {
            const lines = text.split(/\r?\n/).filter(line => line.trim() !== '');

            if (lines.length > 0 && lines[0].toLowerCase().includes('from') && lines[0].toLowerCase().includes('to')) {
                lines.shift();
            }

            const newRoutes: RouteInput[] = lines.map((line, index) => {
                const parts = line.split(',');
                if (parts.length < 3 || !parts[0].trim() || !parts[1].trim() || !parts[2].trim()) {
                    return null;
                }
                const actualDistance = parts[2].trim().replace(/^"|"$/g, '');
                if (isNaN(parseFloat(actualDistance))) return null;

                return {
                    id: Date.now() + index,
                    from: parts[0].trim().replace(/^"|"$/g, ''),
                    to: parts[1].trim().replace(/^"|"$/g, ''),
                    actualDistance,
                };
            }).filter((route): route is RouteInput => route !== null);

            if (newRoutes.length === 0) {
                setFileError("No valid routes found. Ensure format is 'From,To,Total Distance Traveled' with valid numbers.");
                return;
            }

            setRoutes(newRoutes);
            setResults([]);
            setError(null);
        } catch (error) {
            console.error("CSV Parsing Error:", error);
            setFileError(error instanceof Error ? error.message : "Failed to parse the CSV file.");
        }
    };
    reader.onerror = () => {
        setFileError("An error occurred while reading the file.");
    };
    reader.readAsText(file);
  }, []);

  const handleCalculateRoutes = useCallback(async () => {
    const validRoutes = routes.filter(r => 
        r.from.trim() && 
        r.to.trim() && 
        r.actualDistance.trim() && 
        !isNaN(parseFloat(r.actualDistance))
    );
    if (validRoutes.length === 0) {
      setError("Please add at least one valid route with a numeric actual distance.");
      return;
    }

    setIsLoading(true);
    setError(null);
    setFileError(null);
    setResults([]);

    try {
      const apiResults = await getRouteInformation(validRoutes);

      const calculatedResults = apiResults.map((apiResult): RouteResult | null => {
        const originalRoute = validRoutes.find(r => r.from === apiResult.from && r.to === apiResult.to);
        if (!originalRoute) return null;

        const actualDistance = parseFloat(originalRoute.actualDistance);
        const apiDistance = parseFloat(apiResult.totalDistance.replace(/[^0-9.]/g, ''));

        if (isNaN(actualDistance) || isNaN(apiDistance)) return null;

        const difference = actualDistance - apiDistance;
        const deviation = actualDistance !== 0 ? (difference / actualDistance) * 100 : 0;
        const samplingDecision = Math.abs(deviation) > 5 ? 'Selected for Sampling' : 'Not Selected';

        return {
          ...apiResult,
          actualDistance,
          apiDistance,
          difference,
          deviation,
          samplingDecision,
        };
      }).filter((result): result is RouteResult => result !== null);

      setResults(calculatedResults);

    } catch (err) {
      console.error(err);
      setError(err instanceof Error ? err.message : 'An unknown error occurred.');
    } finally {
      setIsLoading(false);
    }
  }, [routes]);

  return (
    <div className="min-h-screen bg-slate-100 font-sans text-slate-800">
      <Header />
      <main className="container mx-auto p-4 md:p-8">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          
          <div className="bg-white p-6 rounded-2xl shadow-lg border border-slate-200">
            <h2 className="text-2xl font-bold text-slate-700 mb-4">Enter Your Routes</h2>
            <p className="text-slate-500 mb-6">Add routes manually, or upload a CSV with "From", "To", and "Total Distance Traveled" columns.</p>
            
            <RouteInputTable 
              routes={routes} 
              onAddRoute={handleAddRoute}
              onRemoveRoute={handleRemoveRoute}
              onRouteChanage={handleRouteChange}
            />

            {fileError && (
              <div className="mt-4 p-3 bg-red-50 text-red-700 text-sm rounded-lg" role="alert">
                  <strong>File Error:</strong> {fileError}
              </div>
            )}
            
            <div className="mt-6 flex flex-col sm:flex-row gap-4 flex-wrap">
              <button 
                onClick={handleAddRoute}
                className="w-full sm:w-auto flex items-center justify-center gap-2 px-5 py-3 text-sm font-medium text-slate-600 bg-slate-100 hover:bg-slate-200 rounded-lg transition-colors duration-200 focus:outline-none focus:ring-2 focus:ring-slate-400 focus:ring-offset-2"
              >
                <PlusIcon />
                Add Another Route
              </button>
              <FileUploadButton onFileSelect={handleFileSelect} />
              <button
                onClick={handleCalculateRoutes}
                disabled={isLoading}
                className="w-full sm:w-auto flex items-center justify-center gap-2 px-6 py-3 text-sm font-semibold text-white bg-blue-600 hover:bg-blue-700 rounded-lg transition-colors duration-200 disabled:bg-blue-400 disabled:cursor-not-allowed focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 shadow-sm"
              >
                {isLoading ? (
                  <>
                    <Loader small={true} />
                    Processing...
                  </>
                ) : (
                  'Calculate & Compare'
                )}
              </button>
            </div>
          </div>

          <div className="bg-white p-6 rounded-2xl shadow-lg border border-slate-200">
            <h2 className="text-2xl font-bold text-slate-700 mb-4">Comparison Results</h2>
            
            {isLoading && (
              <div className="flex flex-col items-center justify-center h-64">
                <Loader />
                <p className="mt-4 text-slate-500">Calculating routes, please wait...</p>
              </div>
            )}

            {error && (
              <div className="flex items-center justify-center h-64 bg-red-50 text-red-700 p-4 rounded-lg">
                <p><strong>Error:</strong> {error}</p>
              </div>
            )}
            
            {!isLoading && !error && results.length === 0 && (
              <div className="flex flex-col items-center justify-center h-64 text-center text-slate-500 bg-slate-50 rounded-lg">
                 <p className="text-lg">Your route comparison will appear here.</p>
                 <p className="mt-1">Enter routes with actual distances and click "Calculate & Compare".</p>
              </div>
            )}

            {!isLoading && results.length > 0 && (
              <ResultsTable results={results} />
            )}
          </div>
        </div>
      </main>
    </div>
  );
};

export default App;
