import React, { useState, useCallback } from 'react';
import type { RouteResult } from '../types';
import { CopyIcon } from './icons/CopyIcon';
import { CheckIcon } from './icons/CheckIcon';
import { DownloadIcon } from './icons/DownloadIcon';

interface ResultsTableProps {
  results: RouteResult[];
}

const ResultsTable: React.FC<ResultsTableProps> = ({ results }) => {
  const [copied, setCopied] = useState(false);

  const formatNumber = (num: number) => num.toFixed(2);

  const downloadCSV = useCallback(() => {
    const headers = [
        'From', 'To', 'Actual Distance (km)', 'API Distance (km)', 'Difference (km)', 
        'Deviation (%)', 'Sampling Decision', 'Estimated Time', 'Route Summary'
    ];
    
    const escapeCSV = (value: string | number) => {
      const stringValue = String(value);
      if (stringValue.includes(',') || stringValue.includes('"') || stringValue.includes('\n')) {
        return `"${stringValue.replace(/"/g, '""')}"`;
      }
      return stringValue;
    };

    const csvContent = [
      headers.join(','),
      ...results.map(r => [
        escapeCSV(r.from),
        escapeCSV(r.to),
        escapeCSV(r.actualDistance),
        escapeCSV(r.apiDistance),
        escapeCSV(formatNumber(r.difference)),
        escapeCSV(formatNumber(r.deviation)),
        escapeCSV(r.samplingDecision),
        escapeCSV(r.estimatedTime),
        escapeCSV(r.routeSummary)
      ].join(','))
    ].join('\n');

    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', 'route_comparison_results.csv');
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  }, [results]);
  
  const copyToClipboard = useCallback(() => {
    const headers = [
        'From', 'To', 'Actual Distance (km)', 'API Distance (km)', 'Difference (km)', 
        'Deviation (%)', 'Sampling Decision', 'Estimated Time', 'Route Summary'
    ].join('\t');
    
    const tsv = results.map(r => 
      [
        r.from, r.to, r.actualDistance, r.apiDistance, formatNumber(r.difference), 
        formatNumber(r.deviation), r.samplingDecision, r.estimatedTime, 
        `"${r.routeSummary.replace(/"/g, '""')}"`
      ].join('\t')
    ).join('\n');
    
    navigator.clipboard.writeText(headers + '\n' + tsv).then(() => {
      setCopied(true);
      setTimeout(() => setCopied(false), 2000);
    });
  }, [results]);

  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center flex-wrap gap-2">
        <p className="text-sm text-slate-500">
          Generated {results.length} result{results.length > 1 ? 's' : ''}.
        </p>
        <div className="flex items-center gap-2">
            <button 
              onClick={copyToClipboard}
              className="flex items-center gap-2 px-4 py-2 text-sm font-medium text-slate-600 bg-slate-100 hover:bg-slate-200 rounded-lg transition-colors duration-200 focus:outline-none focus:ring-2 focus:ring-slate-400 focus:ring-offset-2"
            >
              {copied ? <CheckIcon /> : <CopyIcon />}
              {copied ? 'Copied!' : 'Copy Table'}
            </button>
            <button 
                onClick={downloadCSV}
                className="flex items-center gap-2 px-4 py-2 text-sm font-medium text-slate-600 bg-slate-100 hover:bg-slate-200 rounded-lg transition-colors duration-200 focus:outline-none focus:ring-2 focus:ring-slate-400 focus:ring-offset-2"
            >
                <DownloadIcon />
                Download CSV
            </button>
        </div>
      </div>
      <div className="overflow-x-auto border border-slate-200 rounded-lg">
        <table className="min-w-full divide-y divide-slate-200">
          <thead className="bg-slate-50">
            <tr>
              <th scope="col" className="px-4 py-3 text-left text-xs font-medium text-slate-500 uppercase tracking-wider">From</th>
              <th scope="col" className="px-4 py-3 text-left text-xs font-medium text-slate-500 uppercase tracking-wider">To</th>
              <th scope="col" className="px-4 py-3 text-left text-xs font-medium text-slate-500 uppercase tracking-wider">Actual Dist.</th>
              <th scope="col" className="px-4 py-3 text-left text-xs font-medium text-slate-500 uppercase tracking-wider">API Dist.</th>
              <th scope="col" className="px-4 py-3 text-left text-xs font-medium text-slate-500 uppercase tracking-wider">Difference</th>
              <th scope="col" className="px-4 py-3 text-left text-xs font-medium text-slate-500 uppercase tracking-wider">Deviation</th>
              <th scope="col" className="px-4 py-3 text-left text-xs font-medium text-slate-500 tracking-wider">Selected for Sampling deviation &gt;5%</th>
              <th scope="col" className="px-4 py-3 text-left text-xs font-medium text-slate-500 uppercase tracking-wider">Est. Time</th>
              <th scope="col" className="px-4 py-3 text-left text-xs font-medium text-slate-500 uppercase tracking-wider">Route Summary</th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-slate-200">
            {results.map((result, index) => (
              <tr key={index} className={`${result.samplingDecision === 'Selected for Sampling' ? 'bg-yellow-50 hover:bg-yellow-100' : 'hover:bg-slate-50'} transition-colors`}>
                <td className="px-4 py-4 whitespace-nowrap text-sm font-medium text-slate-900">{result.from}</td>
                <td className="px-4 py-4 whitespace-nowrap text-sm font-medium text-slate-900">{result.to}</td>
                <td className="px-4 py-4 whitespace-nowrap text-sm text-slate-600">{result.actualDistance} km</td>
                <td className="px-4 py-4 whitespace-nowrap text-sm text-slate-600">{result.apiDistance} km</td>
                <td className="px-4 py-4 whitespace-nowrap text-sm text-slate-600">{formatNumber(result.difference)} km</td>
                <td className={`px-4 py-4 whitespace-nowrap text-sm font-medium ${Math.abs(result.deviation) > 5 ? 'text-red-600' : 'text-slate-600'}`}>
                  {formatNumber(result.deviation)}%
                </td>
                <td className="px-4 py-4 whitespace-nowrap text-sm">
                  {result.samplingDecision === 'Selected for Sampling' ? (
                    <span className="px-2.5 py-1 text-xs font-semibold leading-5 rounded-full bg-yellow-100 text-yellow-800">
                      Selected
                    </span>
                  ) : (
                     <span className="text-slate-500">Not Selected</span>
                  )}
                </td>
                <td className="px-4 py-4 whitespace-nowrap text-sm text-slate-600">{result.estimatedTime}</td>
                <td className="px-4 py-4 text-sm text-slate-600 max-w-xs xl:max-w-sm break-words">{result.routeSummary}</td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
};

export default ResultsTable;