
import { GoogleGenAI, Type } from "@google/genai";
import type { RouteInput, ApiRouteResult } from "../types";

const API_KEY = process.env.API_KEY;

if (!API_KEY) {
  throw new Error("API_KEY environment variable is not set.");
}

const ai = new GoogleGenAI({ apiKey: API_KEY });

const systemInstruction = `Role: You are a highly efficient and knowledgeable travel assistant specializing in Indian road networks. Your primary function is to calculate the distance and provide a clear, step-by-step driving route for a four-wheeler between two or more specified locations in India. Your response should be structured, concise, and easy to follow, suitable for a bulk processing and download.

Constraints & Considerations:
- Assume standard road conditions in India, focusing on National Highways (NH), Expressways, and major state roads.
- Your knowledge is based on training data and is not real-time. Do not include live traffic or construction updates.
- The total distance must be in kilometers (km) and the estimated time should be in hours and minutes.
- Prioritize the most direct and logical route for a four-wheeler.
- Be as specific as possible with Indian road names and numbers (e.g., NH 48, Yamuna Expressway).
- Your response must be delivered in the specified JSON format only. Do not include any extra conversational text.`;

const schema = {
  type: Type.ARRAY,
  items: {
    type: Type.OBJECT,
    properties: {
      from: { 
        type: Type.STRING,
        description: 'The starting location of the route.',
      },
      to: { 
        type: Type.STRING,
        description: 'The destination of the route.',
      },
      totalDistance: { 
        type: Type.STRING, 
        description: 'Total distance in kilometers (e.g., "1420 km").' 
      },
      estimatedTime: { 
        type: Type.STRING, 
        description: 'Estimated travel time (e.g., "24 hours 30 minutes").' 
      },
      routeSummary: { 
        type: Type.STRING, 
        description: 'Concise four-wheeler route summary including major highways (e.g., "Via NH 48 and Delhi-Mumbai Expressway").'
      },
    },
    required: ['from', 'to', 'totalDistance', 'estimatedTime', 'routeSummary'],
  },
};


export const getRouteInformation = async (routes: Omit<RouteInput, 'id' | 'actualDistance'>[]): Promise<ApiRouteResult[]> => {
  const model = 'gemini-2.5-flash';

  const userQuery = `
    Please process the following routes and provide the output in the requested JSON format.

    ${routes.map(r => `From: ${r.from}, To: ${r.to}`).join('\n')}
  `;

  try {
    const response = await ai.models.generateContent({
      model: model,
      contents: userQuery,
      config: {
        systemInstruction: systemInstruction,
        responseMimeType: "application/json",
        responseSchema: schema,
      }
    });

    const jsonText = response.text.trim();
    if (!jsonText) {
        throw new Error("Received an empty response from the API.");
    }

    const parsedData = JSON.parse(jsonText);

    // Basic validation to ensure the response is an array
    if (!Array.isArray(parsedData)) {
        throw new Error("API response is not in the expected array format.");
    }
    
    return parsedData as ApiRouteResult[];

  } catch (error) {
    console.error("Error calling Gemini API:", error);
    if (error instanceof Error && error.message.includes('JSON')) {
       throw new Error("Failed to parse the response from the AI. The format was invalid.");
    }
    throw new Error("Failed to fetch route information from the AI service.");
  }
};
