import { useState, useRef } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { Upload, X, FileText } from "lucide-react";
import { ALLOWED_FILE_TYPES, MAX_FILE_SIZE, DOCUMENT_TYPES } from "@/lib/constants";

interface FileUploadProps {
  onUpload: (file: File, docType: string) => Promise<void>;
  isUploading?: boolean;
  allowedTypes?: string[];
  maxSize?: number;
}

export default function FileUpload({ 
  onUpload, 
  isUploading = false,
  allowedTypes = ALLOWED_FILE_TYPES,
  maxSize = MAX_FILE_SIZE
}: FileUploadProps) {
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [docType, setDocType] = useState("");
  const fileInputRef = useRef<HTMLInputElement>(null);
  const { toast } = useToast();

  const validateFile = (file: File): boolean => {
    if (!allowedTypes.includes(file.type)) {
      toast({
        title: "Invalid file type",
        description: `Please select a valid file type. Allowed types: ${allowedTypes.join(", ")}`,
        variant: "destructive",
      });
      return false;
    }

    if (file.size > maxSize) {
      toast({
        title: "File too large",
        description: `File size must be less than ${Math.round(maxSize / 1024 / 1024)}MB`,
        variant: "destructive",
      });
      return false;
    }

    return true;
  };

  const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file && validateFile(file)) {
      setSelectedFile(file);
    }
  };

  const handleUpload = async () => {
    if (!selectedFile || !docType) {
      toast({
        title: "Missing information",
        description: "Please select a file and document type",
        variant: "destructive",
      });
      return;
    }

    try {
      await onUpload(selectedFile, docType);
      setSelectedFile(null);
      setDocType("");
      if (fileInputRef.current) {
        fileInputRef.current.value = "";
      }
      toast({
        title: "Upload successful",
        description: "File uploaded successfully",
      });
    } catch (error) {
      toast({
        title: "Upload failed",
        description: error instanceof Error ? error.message : "Failed to upload file",
        variant: "destructive",
      });
    }
  };

  const handleRemoveFile = () => {
    setSelectedFile(null);
    if (fileInputRef.current) {
      fileInputRef.current.value = "";
    }
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return "0 Bytes";
    const k = 1024;
    const sizes = ["Bytes", "KB", "MB", "GB"];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
  };

  return (
    <div className="space-y-4" data-testid="file-upload">
      <div className="space-y-2">
        <Label htmlFor="file-input">Select File</Label>
        <Input
          id="file-input"
          type="file"
          ref={fileInputRef}
          onChange={handleFileSelect}
          accept={allowedTypes.join(",")}
          className="cursor-pointer"
          data-testid="input-file"
        />
        <p className="text-xs text-muted-foreground">
          Maximum file size: {Math.round(maxSize / 1024 / 1024)}MB. 
          Allowed types: PDF, Excel, CSV, Images
        </p>
      </div>

      {selectedFile && (
        <div className="flex items-center space-x-2 p-3 border border-border rounded-md bg-muted/50">
          <FileText className="h-4 w-4 text-muted-foreground" />
          <div className="flex-1 min-w-0">
            <p className="text-sm font-medium truncate" data-testid="text-selected-filename">
              {selectedFile.name}
            </p>
            <p className="text-xs text-muted-foreground" data-testid="text-selected-filesize">
              {formatFileSize(selectedFile.size)}
            </p>
          </div>
          <Button
            variant="ghost"
            size="icon"
            onClick={handleRemoveFile}
            data-testid="button-remove-file"
          >
            <X className="h-4 w-4" />
          </Button>
        </div>
      )}

      <div className="space-y-2">
        <Label htmlFor="doc-type">Document Type</Label>
        <Select value={docType} onValueChange={setDocType}>
          <SelectTrigger data-testid="select-doc-type">
            <SelectValue placeholder="Select document type" />
          </SelectTrigger>
          <SelectContent>
            {Object.entries(DOCUMENT_TYPES).map(([key, label]) => (
              <SelectItem key={key} value={key}>
                {label}
              </SelectItem>
            ))}
          </SelectContent>
        </Select>
      </div>

      <Button
        onClick={handleUpload}
        disabled={!selectedFile || !docType || isUploading}
        className="w-full"
        data-testid="button-upload"
      >
        {isUploading ? (
          <>
            <Upload className="mr-2 h-4 w-4 animate-spin" />
            Uploading...
          </>
        ) : (
          <>
            <Upload className="mr-2 h-4 w-4" />
            Upload Document
          </>
        )}
      </Button>
    </div>
  );
}
