import { Card, CardContent } from "@/components/ui/card";
import { Skeleton } from "@/components/ui/skeleton";
import { Briefcase, ClipboardCheck, FileCheck, AlertTriangle, TrendingUp, Clock } from "lucide-react";
import type { DashboardStats } from "@/lib/types";

interface KPICardsProps {
  stats?: DashboardStats;
  isLoading: boolean;
}

export default function KPICards({ stats, isLoading }: KPICardsProps) {
  const kpiData = [
    {
      label: "Total Engagements",
      value: stats?.totalEngagements || 0,
      change: "+12%",
      changeType: "positive",
      icon: Briefcase,
      color: "primary"
    },
    {
      label: "Pending Reviews",
      value: stats?.pendingReviews || 0,
      change: "3 overdue",
      changeType: "warning",
      icon: ClipboardCheck,
      color: "orange"
    },
    {
      label: "Filed Returns",
      value: stats?.filedReturns || 0,
      change: "94.5% e-verified",
      changeType: "positive",
      icon: FileCheck,
      color: "green"
    },
    {
      label: "SLA Breaches",
      value: stats?.slaBreaches || 0,
      change: "Requires attention",
      changeType: "negative",
      icon: AlertTriangle,
      color: "red"
    }
  ];

  if (isLoading) {
    return (
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {Array.from({ length: 4 }).map((_, i) => (
          <Card key={i}>
            <CardContent className="p-6">
              <Skeleton className="h-4 w-24 mb-2" />
              <Skeleton className="h-8 w-16 mb-2" />
              <Skeleton className="h-3 w-20" />
            </CardContent>
          </Card>
        ))}
      </div>
    );
  }

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6" data-testid="kpi-cards">
      {kpiData.map((kpi, index) => {
        const Icon = kpi.icon;
        const iconColorClass = {
          primary: "text-primary",
          orange: "text-orange-600",
          green: "text-green-600",
          red: "text-red-600"
        }[kpi.color];

        const iconBgClass = {
          primary: "bg-primary/10",
          orange: "bg-orange-100",
          green: "bg-green-100",
          red: "bg-red-100"
        }[kpi.color];

        const changeIcon = kpi.changeType === "positive" ? TrendingUp : 
                          kpi.changeType === "warning" ? Clock : AlertTriangle;
        const ChangeIcon = changeIcon;

        const changeColorClass = {
          positive: "text-green-600",
          warning: "text-orange-600",
          negative: "text-red-600"
        }[kpi.changeType];

        return (
          <Card key={index} className="shadow-sm" data-testid={`kpi-card-${index}`}>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground" data-testid="text-kpi-label">
                    {kpi.label}
                  </p>
                  <p className="text-2xl font-bold" data-testid="text-kpi-value">
                    {kpi.value}
                  </p>
                  <p className={`text-xs mt-1 flex items-center ${changeColorClass}`}>
                    <ChangeIcon className="mr-1 h-3 w-3" />
                    <span data-testid="text-kpi-change">{kpi.change}</span>
                    {kpi.changeType === "positive" && " from last month"}
                  </p>
                </div>
                <div className={`w-12 h-12 ${iconBgClass} rounded-lg flex items-center justify-center`}>
                  <Icon className={`h-6 w-6 ${iconColorClass}`} />
                </div>
              </div>
            </CardContent>
          </Card>
        );
      })}
    </div>
  );
}
