import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Skeleton } from "@/components/ui/skeleton";
import { Progress } from "@/components/ui/progress";
import type { PipelineStats } from "@/lib/types";
import { ENGAGEMENT_STATUSES, STATUS_COLORS } from "@/lib/constants";

interface PipelineChartProps {
  data?: PipelineStats[];
  isLoading: boolean;
}

export default function PipelineChart({ data, isLoading }: PipelineChartProps) {
  if (isLoading) {
    return (
      <Card>
        <CardHeader>
          <Skeleton className="h-6 w-32" />
        </CardHeader>
        <CardContent className="space-y-3">
          {Array.from({ length: 5 }).map((_, i) => (
            <div key={i} className="flex items-center justify-between">
              <Skeleton className="h-4 w-24" />
              <div className="flex items-center space-x-2">
                <Skeleton className="h-4 w-8" />
                <Skeleton className="h-2 w-24" />
              </div>
            </div>
          ))}
        </CardContent>
      </Card>
    );
  }

  const totalCount = data?.reduce((sum, item) => sum + item.count, 0) || 0;

  // Get the top 5 statuses by count
  const topStatuses = data?.slice().sort((a, b) => b.count - a.count).slice(0, 5) || [];

  const getStatusColor = (status: string) => {
    const colorMap: Record<string, string> = {
      blue: "bg-blue-500",
      indigo: "bg-indigo-500",
      yellow: "bg-yellow-500",
      orange: "bg-orange-500",
      green: "bg-green-500",
      red: "bg-red-500",
      purple: "bg-purple-500",
      pink: "bg-pink-500",
      emerald: "bg-emerald-500",
      gray: "bg-gray-500"
    };
    
    const statusKey = status as keyof typeof STATUS_COLORS;
    const color = STATUS_COLORS[statusKey] || "gray";
    return colorMap[color] || "bg-gray-500";
  };

  return (
    <Card data-testid="pipeline-chart">
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Pipeline Status</CardTitle>
          <Select defaultValue="2024-25">
            <SelectTrigger className="w-32">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="2024-25">AY 2024-25</SelectItem>
              <SelectItem value="2023-24">AY 2023-24</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </CardHeader>
      
      <CardContent>
        <div className="space-y-3">
          {topStatuses.map((item, index) => {
            const percentage = totalCount > 0 ? (item.count / totalCount) * 100 : 0;
            const statusKey = item.status as keyof typeof ENGAGEMENT_STATUSES;
            const displayName = ENGAGEMENT_STATUSES[statusKey] || item.status;
            
            return (
              <div key={item.status} className="flex items-center justify-between" data-testid={`pipeline-item-${index}`}>
                <div className="flex items-center space-x-3">
                  <div className={`w-3 h-3 ${getStatusColor(item.status)} rounded-full`}></div>
                  <span className="text-sm" data-testid="text-status-name">{displayName}</span>
                </div>
                <div className="flex items-center space-x-2">
                  <span className="text-sm font-medium" data-testid="text-status-count">{item.count}</span>
                  <Progress value={percentage} className="w-24 h-2" />
                </div>
              </div>
            );
          })}
        </div>
      </CardContent>
    </Card>
  );
}
