import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { Skeleton } from "@/components/ui/skeleton";
import { Link } from "wouter";
import { Check, Upload, AlertTriangle, MessageCircle } from "lucide-react";
import type { ActivityLogWithActor } from "@/lib/types";

interface RecentActivityProps {
  data?: ActivityLogWithActor[];
  isLoading: boolean;
}

export default function RecentActivity({ data, isLoading }: RecentActivityProps) {
  if (isLoading) {
    return (
      <Card>
        <CardHeader>
          <Skeleton className="h-6 w-32" />
        </CardHeader>
        <CardContent className="space-y-4">
          {Array.from({ length: 4 }).map((_, i) => (
            <div key={i} className="flex items-start space-x-3">
              <Skeleton className="w-8 h-8 rounded-full" />
              <div className="flex-1 space-y-1">
                <Skeleton className="h-4 w-full" />
                <Skeleton className="h-3 w-20" />
              </div>
            </div>
          ))}
        </CardContent>
      </Card>
    );
  }

  const getActivityIcon = (action: string) => {
    switch (action) {
      case "APPROVE_REVIEW":
        return Check;
      case "UPLOAD_DOCUMENT":
        return Upload;
      case "SLA_BREACH":
        return AlertTriangle;
      case "ADD_COMMENT":
        return MessageCircle;
      default:
        return Check;
    }
  };

  const getActivityColor = (action: string) => {
    switch (action) {
      case "APPROVE_REVIEW":
        return "bg-green-100 text-green-600";
      case "UPLOAD_DOCUMENT":
        return "bg-blue-100 text-blue-600";
      case "SLA_BREACH":
        return "bg-red-100 text-red-600";
      case "ADD_COMMENT":
        return "bg-purple-100 text-purple-600";
      default:
        return "bg-gray-100 text-gray-600";
    }
  };

  const formatActivity = (log: ActivityLogWithActor) => {
    const actorName = log.actor?.name || "System";
    
    switch (log.action) {
      case "APPROVE_REVIEW":
        return `${actorName} completed Manager Review for engagement`;
      case "UPLOAD_DOCUMENT":
        return `${actorName} uploaded a document`;
      case "SLA_BREACH":
        return `System flagged SLA breach for engagement`;
      case "ADD_COMMENT":
        return `${actorName} mentioned someone in computation notes`;
      default:
        return `${actorName} performed ${log.action}`;
    }
  };

  const getUserInitials = (name: string) => {
    return name
      .split(" ")
      .map(n => n[0])
      .join("")
      .toUpperCase()
      .substring(0, 2);
  };

  const formatTimeAgo = (timestamp: string | Date) => {
    const date = new Date(timestamp);
    const now = new Date();
    const diffInMinutes = Math.floor((now.getTime() - date.getTime()) / (1000 * 60));
    
    if (diffInMinutes < 60) {
      return `${diffInMinutes} minutes ago`;
    } else if (diffInMinutes < 1440) {
      return `${Math.floor(diffInMinutes / 60)} hours ago`;
    } else {
      return `${Math.floor(diffInMinutes / 1440)} days ago`;
    }
  };

  return (
    <Card data-testid="recent-activity">
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Recent Activity</CardTitle>
          <Link href="/audit" className="text-sm text-primary hover:underline" data-testid="link-view-all">
            View All
          </Link>
        </div>
      </CardHeader>
      
      <CardContent>
        <div className="space-y-4">
          {data?.map((log, index) => {
            const Icon = getActivityIcon(log.action);
            const actorName = log.actor?.name || "System";
            
            return (
              <div key={log.id} className="flex items-start space-x-3" data-testid={`activity-item-${index}`}>
                <div className={`w-8 h-8 rounded-full flex items-center justify-center flex-shrink-0 ${getActivityColor(log.action)}`}>
                  <Icon className="h-4 w-4" />
                </div>
                <div className="flex-1 min-w-0">
                  <p className="text-sm" data-testid="text-activity-description">
                    {formatActivity(log)}
                  </p>
                  <p className="text-xs text-muted-foreground" data-testid="text-activity-time">
                    {log.createdAt ? formatTimeAgo(log.createdAt) : "Unknown time"}
                  </p>
                </div>
              </div>
            );
          })}
          
          {(!data || data.length === 0) && (
            <div className="text-center py-4">
              <p className="text-sm text-muted-foreground">No recent activity</p>
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
}
