import { useState } from "react";
import { useUpdateChecklistItem } from "@/hooks/use-engagements";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Checkbox } from "@/components/ui/checkbox";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";
import { ChevronDown, ChevronRight, User, Calendar, AlertTriangle } from "lucide-react";
import type { ChecklistItem } from "@shared/schema";
import { CHECKLIST_STATUS } from "@/lib/constants";

interface ChecklistProps {
  engagementId: string;
  items: ChecklistItem[];
  progressPercentage: number;
  completedItems: number;
  totalItems: number;
}

export default function Checklist({ 
  engagementId, 
  items, 
  progressPercentage, 
  completedItems, 
  totalItems 
}: ChecklistProps) {
  const [expandedItems, setExpandedItems] = useState<Set<string>>(new Set());
  const [editingItem, setEditingItem] = useState<string | null>(null);
  const [notes, setNotes] = useState<string>("");
  const updateChecklistItem = useUpdateChecklistItem();

  const toggleExpanded = (itemId: string) => {
    const newExpanded = new Set(expandedItems);
    if (newExpanded.has(itemId)) {
      newExpanded.delete(itemId);
    } else {
      newExpanded.add(itemId);
    }
    setExpandedItems(newExpanded);
  };

  const handleStatusChange = async (itemId: string, status: string) => {
    try {
      await updateChecklistItem.mutateAsync({
        id: itemId,
        data: {
          status: status as any,
          completedAt: status === "DONE" ? new Date() : undefined
        }
      });
    } catch (error) {
      console.error("Failed to update checklist item:", error);
    }
  };

  const handleNotesUpdate = async (itemId: string) => {
    try {
      await updateChecklistItem.mutateAsync({
        id: itemId,
        data: { notes }
      });
      setEditingItem(null);
      setNotes("");
    } catch (error) {
      console.error("Failed to update notes:", error);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "DONE":
        return "bg-green-100 text-green-800";
      case "IN_PROGRESS":
        return "bg-blue-100 text-blue-800";
      case "PENDING":
        return "bg-gray-100 text-gray-800";
      case "NA":
        return "bg-purple-100 text-purple-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getPriorityIcon = (mandatory: boolean) => {
    if (mandatory) {
      return <AlertTriangle className="h-4 w-4 text-red-500" />;
    }
    return null;
  };

  // Group items by section for better organization
  const groupedItems = items.reduce((acc, item) => {
    const section = item.sectionRef || "General";
    if (!acc[section]) {
      acc[section] = [];
    }
    acc[section].push(item);
    return acc;
  }, {} as Record<string, ChecklistItem[]>);

  return (
    <Card data-testid="checklist">
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>ITR Checklist</CardTitle>
          <div className="flex items-center space-x-4">
            <div className="text-right">
              <p className="text-sm text-muted-foreground">Progress</p>
              <p className="font-semibold" data-testid="text-progress">
                {completedItems}/{totalItems} Complete
              </p>
            </div>
            <div className="w-32">
              <Progress value={progressPercentage} className="h-2" />
            </div>
          </div>
        </div>
      </CardHeader>
      
      <CardContent>
        <div className="space-y-6">
          {Object.entries(groupedItems).map(([section, sectionItems]) => (
            <div key={section} className="space-y-2">
              <h4 className="font-medium text-sm text-muted-foreground uppercase tracking-wide" data-testid={`section-${section}`}>
                {section}
              </h4>
              
              <div className="space-y-2">
                {sectionItems.map((item) => {
                  const isExpanded = expandedItems.has(item.id);
                  const isEditing = editingItem === item.id;
                  
                  return (
                    <div key={item.id} className="border border-border rounded-lg" data-testid={`checklist-item-${item.id}`}>
                      <div className="flex items-center space-x-3 p-3">
                        <Checkbox 
                          checked={item.status === "DONE"}
                          onCheckedChange={(checked) => 
                            handleStatusChange(item.id, checked ? "DONE" : "PENDING")
                          }
                          data-testid="checkbox-item-status"
                        />
                        
                        <div className="flex-1 min-w-0">
                          <div className="flex items-center space-x-2">
                            {getPriorityIcon(item.mandatory)}
                            <p className={`text-sm font-medium ${item.status === "DONE" ? "line-through text-muted-foreground" : ""}`} data-testid="text-item-description">
                              {item.item}
                            </p>
                            {item.mandatory && (
                              <Badge variant="outline" className="text-xs" data-testid="badge-mandatory">
                                Required
                              </Badge>
                            )}
                          </div>
                          
                          <div className="flex items-center space-x-4 mt-1">
                            <Badge className={getStatusColor(item.status)} data-testid="badge-item-status">
                              {CHECKLIST_STATUS[item.status as keyof typeof CHECKLIST_STATUS]}
                            </Badge>
                            
                            {item.sectionRef && (
                              <span className="text-xs text-muted-foreground" data-testid="text-section-ref">
                                {item.sectionRef}
                              </span>
                            )}
                            
                            {item.assigneeId && (
                              <div className="flex items-center space-x-1 text-xs text-muted-foreground">
                                <User className="h-3 w-3" />
                                <span data-testid="text-assignee">Assigned</span>
                              </div>
                            )}
                            
                            {item.dueAt && (
                              <div className="flex items-center space-x-1 text-xs text-muted-foreground">
                                <Calendar className="h-3 w-3" />
                                <span data-testid="text-due-date">
                                  {new Date(item.dueAt).toLocaleDateString()}
                                </span>
                              </div>
                            )}
                          </div>
                        </div>
                        
                        <div className="flex items-center space-x-2">
                          <Select 
                            value={item.status} 
                            onValueChange={(value) => handleStatusChange(item.id, value)}
                          >
                            <SelectTrigger className="w-32 h-8" data-testid="select-item-status">
                              <SelectValue />
                            </SelectTrigger>
                            <SelectContent>
                              <SelectItem value="PENDING">Pending</SelectItem>
                              <SelectItem value="IN_PROGRESS">In Progress</SelectItem>
                              <SelectItem value="DONE">Done</SelectItem>
                              <SelectItem value="NA">N/A</SelectItem>
                            </SelectContent>
                          </Select>
                          
                          <Collapsible open={isExpanded} onOpenChange={() => toggleExpanded(item.id)}>
                            <CollapsibleTrigger asChild>
                              <Button variant="ghost" size="icon" data-testid="button-toggle-details">
                                {isExpanded ? (
                                  <ChevronDown className="h-4 w-4" />
                                ) : (
                                  <ChevronRight className="h-4 w-4" />
                                )}
                              </Button>
                            </CollapsibleTrigger>
                          </Collapsible>
                        </div>
                      </div>
                      
                      <Collapsible open={isExpanded} onOpenChange={() => toggleExpanded(item.id)}>
                        <CollapsibleContent>
                          <div className="px-3 pb-3 border-t border-border mt-3 pt-3">
                            <div className="space-y-3">
                              {/* Notes Section */}
                              <div>
                                <div className="flex items-center justify-between mb-2">
                                  <label className="text-sm font-medium">Notes</label>
                                  {!isEditing && (
                                    <Button 
                                      variant="ghost" 
                                      size="sm"
                                      onClick={() => {
                                        setEditingItem(item.id);
                                        setNotes(item.notes || "");
                                      }}
                                      data-testid="button-edit-notes"
                                    >
                                      Edit
                                    </Button>
                                  )}
                                </div>
                                
                                {isEditing ? (
                                  <div className="space-y-2">
                                    <Textarea
                                      value={notes}
                                      onChange={(e) => setNotes(e.target.value)}
                                      placeholder="Add notes for this checklist item..."
                                      rows={3}
                                      data-testid="textarea-notes"
                                    />
                                    <div className="flex space-x-2">
                                      <Button 
                                        size="sm" 
                                        onClick={() => handleNotesUpdate(item.id)}
                                        data-testid="button-save-notes"
                                      >
                                        Save
                                      </Button>
                                      <Button 
                                        variant="outline" 
                                        size="sm"
                                        onClick={() => {
                                          setEditingItem(null);
                                          setNotes("");
                                        }}
                                        data-testid="button-cancel-notes"
                                      >
                                        Cancel
                                      </Button>
                                    </div>
                                  </div>
                                ) : (
                                  <div className="text-sm text-muted-foreground" data-testid="text-notes">
                                    {item.notes || "No notes added"}
                                  </div>
                                )}
                              </div>
                              
                              {/* Timeline */}
                              {item.completedAt && (
                                <div className="text-xs text-muted-foreground" data-testid="text-completion-info">
                                  Completed on {new Date(item.completedAt).toLocaleDateString()} at {new Date(item.completedAt).toLocaleTimeString()}
                                </div>
                              )}
                            </div>
                          </div>
                        </CollapsibleContent>
                      </Collapsible>
                    </div>
                  );
                })}
              </div>
            </div>
          ))}
          
          {items.length === 0 && (
            <div className="text-center py-8">
              <AlertTriangle className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
              <p className="text-lg font-medium mb-2">No checklist items</p>
              <p className="text-muted-foreground">Checklist items will be auto-generated based on ITR form type</p>
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
}
