import { useState } from "react";
import { useCreateComment } from "@/hooks/use-engagements";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { Badge } from "@/components/ui/badge";
import { MessageCircle, Send, AtSign } from "lucide-react";
import type { CommentWithAuthor } from "@/lib/types";

interface CommentsPanelProps {
  engagementId: string;
  comments: CommentWithAuthor[];
}

export default function CommentsPanel({ engagementId, comments }: CommentsPanelProps) {
  const [newComment, setNewComment] = useState("");
  const [mentions, setMentions] = useState<string[]>([]);
  const createComment = useCreateComment();

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!newComment.trim()) return;

    try {
      await createComment.mutateAsync({
        engagementId,
        entityType: "ENGAGEMENT",
        entityId: engagementId,
        body: newComment,
        mentions
      });
      
      setNewComment("");
      setMentions([]);
    } catch (error) {
      console.error("Failed to create comment:", error);
    }
  };

  const getUserInitials = (name: string) => {
    return name
      .split(" ")
      .map(n => n[0])
      .join("")
      .toUpperCase()
      .substring(0, 2);
  };

  const formatTimeAgo = (timestamp: string | Date) => {
    const date = new Date(timestamp);
    const now = new Date();
    const diffInMinutes = Math.floor((now.getTime() - date.getTime()) / (1000 * 60));
    
    if (diffInMinutes < 1) return "Just now";
    if (diffInMinutes < 60) return `${diffInMinutes} minutes ago`;
    
    const diffInHours = Math.floor(diffInMinutes / 60);
    if (diffInHours < 24) return `${diffInHours} hours ago`;
    
    const diffInDays = Math.floor(diffInHours / 24);
    return `${diffInDays} days ago`;
  };

  const extractMentions = (text: string): string[] => {
    const mentionRegex = /@(\w+)/g;
    const matches = text.match(mentionRegex);
    return matches ? matches.map(match => match.substring(1)) : [];
  };

  const handleCommentChange = (value: string) => {
    setNewComment(value);
    setMentions(extractMentions(value));
  };

  const renderCommentText = (text: string) => {
    // Simple mention highlighting - in a real app you'd want more sophisticated parsing
    return text.split(/(@\w+)/g).map((part, index) => {
      if (part.startsWith('@')) {
        return (
          <span key={index} className="text-primary font-medium" data-testid="text-mention">
            {part}
          </span>
        );
      }
      return part;
    });
  };

  return (
    <Card data-testid="comments-panel">
      <CardHeader>
        <CardTitle className="flex items-center space-x-2">
          <MessageCircle className="h-5 w-5" />
          <span>Comments ({comments.length})</span>
        </CardTitle>
      </CardHeader>
      
      <CardContent>
        <div className="space-y-6">
          {/* Comments List */}
          <div className="space-y-4 max-h-96 overflow-y-auto">
            {comments.length > 0 ? (
              comments.map((comment) => (
                <div key={comment.id} className="flex items-start space-x-3" data-testid={`comment-${comment.id}`}>
                  <Avatar className="w-8 h-8">
                    <AvatarFallback className="bg-secondary text-secondary-foreground text-xs">
                      {getUserInitials(comment.author.name)}
                    </AvatarFallback>
                  </Avatar>
                  
                  <div className="flex-1 min-w-0">
                    <div className="bg-muted rounded-lg p-3">
                      <div className="flex items-center justify-between mb-1">
                        <span className="text-sm font-medium" data-testid="text-author-name">
                          {comment.author.name}
                        </span>
                        <div className="flex items-center space-x-2">
                          <Badge variant="outline" className="text-xs" data-testid="badge-author-role">
                            {comment.author.role}
                          </Badge>
                          <span className="text-xs text-muted-foreground" data-testid="text-comment-time">
                            {comment.createdAt ? formatTimeAgo(comment.createdAt) : "Unknown time"}
                          </span>
                        </div>
                      </div>
                      
                      <p className="text-sm" data-testid="text-comment-body">
                        {renderCommentText(comment.body)}
                      </p>
                      
                      {comment.mentions && comment.mentions.length > 0 && (
                        <div className="flex items-center space-x-1 mt-2">
                          <AtSign className="h-3 w-3 text-muted-foreground" />
                          <span className="text-xs text-muted-foreground" data-testid="text-mentions">
                            {comment.mentions.length} mention{comment.mentions.length > 1 ? 's' : ''}
                          </span>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              ))
            ) : (
              <div className="text-center py-8">
                <MessageCircle className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
                <p className="text-lg font-medium mb-2">No comments yet</p>
                <p className="text-muted-foreground">Start the conversation by adding a comment</p>
              </div>
            )}
          </div>
          
          {/* Add Comment Form */}
          <form onSubmit={handleSubmit} className="space-y-3">
            <div className="flex items-start space-x-3">
              <Avatar className="w-8 h-8">
                <AvatarFallback className="bg-secondary text-secondary-foreground text-xs">
                  You
                </AvatarFallback>
              </Avatar>
              
              <div className="flex-1 space-y-2">
                <Textarea
                  value={newComment}
                  onChange={(e) => handleCommentChange(e.target.value)}
                  placeholder="Add a comment... Use @username to mention team members"
                  rows={3}
                  className="resize-none"
                  data-testid="textarea-new-comment"
                />
                
                {mentions.length > 0 && (
                  <div className="flex items-center space-x-2">
                    <AtSign className="h-4 w-4 text-muted-foreground" />
                    <div className="flex space-x-1">
                      {mentions.map((mention, index) => (
                        <Badge key={index} variant="outline" className="text-xs" data-testid={`badge-mention-${index}`}>
                          @{mention}
                        </Badge>
                      ))}
                    </div>
                  </div>
                )}
                
                <div className="flex justify-end">
                  <Button 
                    type="submit" 
                    disabled={!newComment.trim() || createComment.isPending}
                    data-testid="button-add-comment"
                  >
                    <Send className="mr-2 h-4 w-4" />
                    {createComment.isPending ? "Posting..." : "Comment"}
                  </Button>
                </div>
              </div>
            </div>
            
            <div className="text-xs text-muted-foreground pl-11">
              <p>💡 Pro tip: Use @username to mention team members and notify them about updates</p>
            </div>
          </form>
        </div>
      </CardContent>
    </Card>
  );
}
