import { useState } from "react";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import FileUpload from "@/components/common/file-upload";
import { Download, Eye, Trash2, Upload, FileText, FileX, MoreVertical } from "lucide-react";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import type { Document } from "@shared/schema";
import { DOCUMENT_TYPES } from "@/lib/constants";

interface DocumentsPanelProps {
  engagementId: string;
  documents: Document[];
}

export default function DocumentsPanel({ engagementId, documents }: DocumentsPanelProps) {
  const [isUploadDialogOpen, setIsUploadDialogOpen] = useState(false);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const uploadDocument = useMutation({
    mutationFn: async ({ file, docType }: { file: File; docType: string }) => {
      const formData = new FormData();
      formData.append('file', file);
      formData.append('docType', docType);

      const response = await fetch(`/api/engagements/${engagementId}/documents`, {
        method: 'POST',
        headers: {
          Authorization: `Bearer ${localStorage.getItem("accessToken")}`,
        },
        body: formData,
      });

      if (!response.ok) {
        throw new Error('Upload failed');
      }

      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/engagements", engagementId, "documents"] });
      setIsUploadDialogOpen(false);
      toast({
        title: "Upload successful",
        description: "Document uploaded successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Upload failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const deleteDocument = useMutation({
    mutationFn: async (documentId: string) => {
      const response = await apiRequest("DELETE", `/api/documents/${documentId}`);
      return response;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/engagements", engagementId, "documents"] });
      toast({
        title: "Document deleted",
        description: "Document deleted successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Delete failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleUpload = async (file: File, docType: string) => {
    await uploadDocument.mutateAsync({ file, docType });
  };

  const handleDownload = async (documentId: string, fileName: string) => {
    try {
      const response = await fetch(`/api/documents/${documentId}/download`, {
        headers: {
          Authorization: `Bearer ${localStorage.getItem("accessToken")}`,
        },
      });

      if (response.ok) {
        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = fileName;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
      } else {
        throw new Error('Download failed');
      }
    } catch (error) {
      toast({
        title: "Download failed",
        description: "Failed to download document",
        variant: "destructive",
      });
    }
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return "0 Bytes";
    const k = 1024;
    const sizes = ["Bytes", "KB", "MB", "GB"];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
  };

  const getFileIcon = (docType: string) => {
    const iconMap: Record<string, string> = {
      "FORM_16": "📄",
      "AIS_26AS": "📊",
      "TIS": "📋",
      "BANK_STATEMENT": "🏦",
      "CAPITAL_GAINS": "📈",
      "LOAN_INTEREST": "🏠",
      "DONATION": "💝",
      "HOUSE_PROPERTY": "🏘️",
      "BUSINESS": "💼",
      "AUDIT_REPORT": "🔍",
      "GENERAL": "📄"
    };
    return iconMap[docType] || "📄";
  };

  const getDocumentTypeColor = (docType: string) => {
    const colorMap: Record<string, string> = {
      "FORM_16": "bg-blue-100 text-blue-800",
      "AIS_26AS": "bg-green-100 text-green-800",
      "TIS": "bg-purple-100 text-purple-800",
      "BANK_STATEMENT": "bg-orange-100 text-orange-800",
      "CAPITAL_GAINS": "bg-emerald-100 text-emerald-800",
      "AUDIT_REPORT": "bg-red-100 text-red-800",
      "GENERAL": "bg-gray-100 text-gray-800"
    };
    return colorMap[docType] || "bg-gray-100 text-gray-800";
  };

  return (
    <Card data-testid="documents-panel">
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Documents ({documents.length})</CardTitle>
          <Dialog open={isUploadDialogOpen} onOpenChange={setIsUploadDialogOpen}>
            <DialogTrigger asChild>
              <Button data-testid="button-upload-document">
                <Upload className="mr-2 h-4 w-4" />
                Upload
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-lg">
              <DialogHeader>
                <DialogTitle>Upload Document</DialogTitle>
              </DialogHeader>
              <FileUpload 
                onUpload={handleUpload}
                isUploading={uploadDocument.isPending}
              />
            </DialogContent>
          </Dialog>
        </div>
      </CardHeader>
      
      <CardContent>
        {documents.length > 0 ? (
          <div className="space-y-3">
            {documents.map((document) => (
              <div 
                key={document.id} 
                className="flex items-center space-x-3 p-3 border border-border rounded-md hover:bg-muted/50 transition-colors"
                data-testid={`document-item-${document.id}`}
              >
                <div className="text-2xl flex-shrink-0">
                  {getFileIcon(document.docType)}
                </div>
                
                <div className="flex-1 min-w-0">
                  <div className="flex items-center space-x-2 mb-1">
                    <p className="text-sm font-medium truncate" data-testid="text-filename">
                      {document.fileName}
                    </p>
                    <Badge 
                      className={`text-xs ${getDocumentTypeColor(document.docType)}`}
                      data-testid="badge-doc-type"
                    >
                      {DOCUMENT_TYPES[document.docType as keyof typeof DOCUMENT_TYPES] || document.docType}
                    </Badge>
                  </div>
                  
                  <div className="flex items-center space-x-4 text-xs text-muted-foreground">
                    <span data-testid="text-file-size">{formatFileSize(document.fileSize)}</span>
                    <span data-testid="text-upload-date">
                      {document.createdAt ? new Date(document.createdAt).toLocaleDateString() : "Unknown date"}
                    </span>
                    <span data-testid="text-version">v{document.version}</span>
                    {document.clientVisible && (
                      <Badge variant="outline" className="text-xs" data-testid="badge-client-visible">
                        Client Visible
                      </Badge>
                    )}
                  </div>
                </div>
                
                <div className="flex items-center space-x-1">
                  <Button 
                    variant="ghost" 
                    size="icon"
                    onClick={() => handleDownload(document.id, document.fileName)}
                    data-testid="button-download"
                  >
                    <Download className="h-4 w-4" />
                  </Button>
                  
                  <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                      <Button variant="ghost" size="icon" data-testid="button-document-menu">
                        <MoreVertical className="h-4 w-4" />
                      </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent align="end">
                      <DropdownMenuItem 
                        onClick={() => handleDownload(document.id, document.fileName)}
                        data-testid="menu-download"
                      >
                        <Download className="mr-2 h-4 w-4" />
                        Download
                      </DropdownMenuItem>
                      <DropdownMenuItem 
                        onClick={() => handleDownload(document.id, document.fileName)}
                        data-testid="menu-view"
                      >
                        <Eye className="mr-2 h-4 w-4" />
                        View
                      </DropdownMenuItem>
                      <DropdownMenuItem 
                        className="text-red-600"
                        onClick={() => deleteDocument.mutate(document.id)}
                        disabled={deleteDocument.isPending}
                        data-testid="menu-delete"
                      >
                        <Trash2 className="mr-2 h-4 w-4" />
                        Delete
                      </DropdownMenuItem>
                    </DropdownMenuContent>
                  </DropdownMenu>
                </div>
              </div>
            ))}
          </div>
        ) : (
          <div className="text-center py-8">
            <FileX className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
            <p className="text-lg font-medium mb-2">No documents uploaded</p>
            <p className="text-muted-foreground mb-4">Upload documents to get started</p>
            <Dialog open={isUploadDialogOpen} onOpenChange={setIsUploadDialogOpen}>
              <DialogTrigger asChild>
                <Button data-testid="button-upload-first">
                  <Upload className="mr-2 h-4 w-4" />
                  Upload First Document
                </Button>
              </DialogTrigger>
              <DialogContent className="max-w-lg">
                <DialogHeader>
                  <DialogTitle>Upload Document</DialogTitle>
                </DialogHeader>
                <FileUpload 
                  onUpload={handleUpload}
                  isUploading={uploadDocument.isPending}
                />
              </DialogContent>
            </Dialog>
          </div>
        )}
      </CardContent>
    </Card>
  );
}
