import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Check, Clock, AlertCircle } from "lucide-react";
import { ENGAGEMENT_STATUSES } from "@/lib/constants";
import type { Engagement } from "@shared/schema";

interface WorkflowProgressProps {
  engagement: Engagement;
  onStatusChange: (newStatus: string, reason?: string) => void;
}

const WORKFLOW_STAGES = [
  "DATA_REQUESTED",
  "DATA_RECEIVED_PRELIMINARY",
  "KYC_CONSENT_CAPTURED",
  "DOC_INDEXING_VETTING",
  "AIS_TIS_IMPORTED_RECONCILED",
  "COMPUTATION_PREPARED",
  "MANAGER_REVIEW",
  "PARTNER_REVIEW",
  "CLIENT_REVIEW_SIGNOFF",
  "ITR_CREATED_JSON_VALIDATED",
  "ITR_FILED_ACK_CAPTURED",
  "E_VERIFICATION_PENDING",
  "E_VERIFICATION_DONE",
  "CPC_PROCESSING",
  "REFUND_ADJUSTMENT_UPDATE",
  "CLOSED"
];

export default function WorkflowProgress({ engagement, onStatusChange }: WorkflowProgressProps) {
  const currentStageIndex = WORKFLOW_STAGES.indexOf(engagement.status);

  const getStageStatus = (index: number) => {
    if (index < currentStageIndex) return "completed";
    if (index === currentStageIndex) return "current";
    return "pending";
  };

  const getStageIcon = (status: string) => {
    switch (status) {
      case "completed":
        return <Check className="h-4 w-4 text-white" />;
      case "current":
        return <Clock className="h-4 w-4 text-white" />;
      default:
        return <span className="text-gray-600 text-xs">{WORKFLOW_STAGES.indexOf(engagement.status) + 1}</span>;
    }
  };

  const getStageColorClass = (status: string) => {
    switch (status) {
      case "completed":
        return "bg-green-500";
      case "current":
        return "bg-yellow-500";
      default:
        return "bg-gray-300";
    }
  };

  const canAdvanceToNext = () => {
    return currentStageIndex < WORKFLOW_STAGES.length - 1;
  };

  const getNextStage = () => {
    if (canAdvanceToNext()) {
      return WORKFLOW_STAGES[currentStageIndex + 1];
    }
    return null;
  };

  return (
    <Card data-testid="workflow-progress">
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Workflow Progress</CardTitle>
          <Badge variant="outline" data-testid="badge-current-stage">
            Stage {currentStageIndex + 1} of {WORKFLOW_STAGES.length}
          </Badge>
        </div>
      </CardHeader>
      
      <CardContent>
        <div className="space-y-6">
          {/* Progress Visualization */}
          <div className="relative">
            <div className="absolute left-4 top-8 bottom-8 w-px bg-border"></div>
            
            <div className="space-y-4">
              {WORKFLOW_STAGES.slice(0, 8).map((stage, index) => {
                const status = getStageStatus(index);
                const stageKey = stage as keyof typeof ENGAGEMENT_STATUSES;
                const displayName = ENGAGEMENT_STATUSES[stageKey] || stage;
                
                return (
                  <div key={stage} className="relative flex items-center space-x-4" data-testid={`workflow-stage-${index}`}>
                    <div className={`w-8 h-8 ${getStageColorClass(status)} rounded-full flex items-center justify-center flex-shrink-0 z-10`}>
                      {getStageIcon(status)}
                    </div>
                    <div className="flex-1">
                      <p className={`text-sm font-medium ${status === "current" ? "text-yellow-700" : status === "completed" ? "text-green-700" : "text-muted-foreground"}`}>
                        {displayName}
                      </p>
                      {status === "completed" && (
                        <p className="text-xs text-muted-foreground" data-testid="text-completion-info">
                          Completed • {/* This would show completion time and user */}
                        </p>
                      )}
                      {status === "current" && (
                        <p className="text-xs text-yellow-600" data-testid="text-current-info">
                          In progress • Due in 18h
                        </p>
                      )}
                      {status === "pending" && (
                        <p className="text-xs text-muted-foreground" data-testid="text-pending-info">
                          Pending
                        </p>
                      )}
                    </div>
                  </div>
                );
              })}
            </div>
          </div>

          {/* Action Buttons */}
          <div className="flex space-x-3 pt-4 border-t border-border">
            {canAdvanceToNext() && (
              <Button 
                onClick={() => onStatusChange(getNextStage()!)}
                data-testid="button-advance-stage"
              >
                Advance to Next Stage
              </Button>
            )}
            
            {engagement.status === "COMPUTATION_PREPARED" && (
              <Button 
                onClick={() => onStatusChange("MANAGER_REVIEW")}
                data-testid="button-submit-review"
              >
                Submit for Review
              </Button>
            )}
            
            {(engagement.status === "MANAGER_REVIEW" || engagement.status === "PARTNER_REVIEW") && (
              <div className="flex space-x-2">
                <Button 
                  className="bg-green-500 hover:bg-green-600"
                  onClick={() => onStatusChange(getNextStage()!)}
                  data-testid="button-approve"
                >
                  <Check className="mr-2 h-4 w-4" />
                  Approve
                </Button>
                <Button 
                  variant="outline"
                  className="border-orange-500 text-orange-600 hover:bg-orange-50"
                  onClick={() => onStatusChange("COMPUTATION_PREPARED", "Changes requested")}
                  data-testid="button-request-changes"
                >
                  <AlertCircle className="mr-2 h-4 w-4" />
                  Request Changes
                </Button>
              </div>
            )}
          </div>
        </div>
      </CardContent>
    </Card>
  );
}
