import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { useClients } from "@/hooks/use-clients";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Switch } from "@/components/ui/switch";
import { Skeleton } from "@/components/ui/skeleton";
import { Loader2 } from "lucide-react";
import type { InsertEngagement } from "@shared/schema";
import { ITR_FORM_TYPES, TIER_TYPES, PRIORITY_LEVELS } from "@/lib/constants";

const engagementSchema = z.object({
  clientId: z.string().min(1, "Client is required"),
  ay: z.string().regex(/^\d{4}-\d{2}$/, "Assessment Year must be in format YYYY-YY (e.g., 2024-25)"),
  itrFormType: z.enum(["ITR-1", "ITR-2", "ITR-3", "ITR-4", "ITR-5", "ITR-6", "ITR-7"]),
  tier: z.enum(["INDIVIDUAL", "HUF", "FIRM", "COMPANY", "TRUST"]),
  isAuditApplicable: z.boolean().default(false),
  priority: z.enum(["LOW", "MEDIUM", "HIGH", "CRITICAL"]).default("MEDIUM"),
  filingDueDate: z.string().optional(),
  auditDueDate: z.string().optional()
});

type EngagementFormData = z.infer<typeof engagementSchema>;

interface EngagementFormProps {
  onSubmit: (data: InsertEngagement) => Promise<void>;
  initialData?: Partial<EngagementFormData>;
  isLoading?: boolean;
}

export default function EngagementForm({ onSubmit, initialData, isLoading = false }: EngagementFormProps) {
  const { data: clientsData, isLoading: clientsLoading } = useClients({ limit: 100 });
  
  const form = useForm<EngagementFormData>({
    resolver: zodResolver(engagementSchema),
    defaultValues: {
      clientId: initialData?.clientId || "",
      ay: initialData?.ay || "2024-25",
      itrFormType: initialData?.itrFormType || "ITR-1",
      tier: initialData?.tier || "INDIVIDUAL",
      isAuditApplicable: initialData?.isAuditApplicable || false,
      priority: initialData?.priority || "MEDIUM",
      filingDueDate: initialData?.filingDueDate || "",
      auditDueDate: initialData?.auditDueDate || ""
    }
  });

  const watchedTier = form.watch("tier");
  const watchedItrForm = form.watch("itrFormType");

  const handleSubmit = async (data: EngagementFormData) => {
    const submitData: InsertEngagement = {
      clientId: data.clientId,
      ay: data.ay,
      itrFormType: data.itrFormType,
      tier: data.tier,
      isAuditApplicable: data.isAuditApplicable,
      priority: data.priority,
      dueDates: {
        filingDue: data.filingDueDate ? new Date(data.filingDueDate).toISOString() : undefined,
        auditDue: data.auditDueDate ? new Date(data.auditDueDate).toISOString() : undefined
      }
    };
    
    await onSubmit(submitData);
  };

  // Get default due dates based on tier and form type
  const getDefaultDueDates = () => {
    const ayYear = parseInt(form.getValues("ay").split("-")[0]);
    const nextYear = ayYear + 1;
    
    // Default due dates based on ITR form type and tier
    const dueDates: Record<string, { filing: string; audit?: string }> = {
      "ITR-1": { filing: `${nextYear}-07-31` },
      "ITR-2": { filing: `${nextYear}-07-31` },
      "ITR-3": { filing: `${nextYear}-07-31` },
      "ITR-4": { filing: `${nextYear}-07-31` },
      "ITR-5": { filing: `${nextYear}-10-31` },
      "ITR-6": { filing: `${nextYear}-10-31` },
      "ITR-7": { filing: `${nextYear}-10-31` }
    };

    return dueDates[watchedItrForm] || { filing: `${nextYear}-07-31` };
  };

  // Set default due dates when ITR form type changes
  const handleITRFormChange = (value: string) => {
    const defaultDates = getDefaultDueDates();
    form.setValue("filingDueDate", defaultDates.filing);
    if (defaultDates.audit) {
      form.setValue("auditDueDate", defaultDates.audit);
    }
  };

  if (clientsLoading) {
    return (
      <Card>
        <CardHeader>
          <Skeleton className="h-6 w-48" />
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {Array.from({ length: 5 }).map((_, i) => (
              <Skeleton key={i} className="h-10 w-full" />
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(handleSubmit)} className="space-y-6" data-testid="engagement-form">
        {/* Basic Information */}
        <Card>
          <CardHeader>
            <CardTitle>Basic Information</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <FormField
                control={form.control}
                name="clientId"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Client *</FormLabel>
                    <Select onValueChange={field.onChange} defaultValue={field.value}>
                      <FormControl>
                        <SelectTrigger data-testid="select-client">
                          <SelectValue placeholder="Select a client" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        {clientsData?.data.map((client) => (
                          <SelectItem key={client.id} value={client.id}>
                            {client.displayName} ({client.pan})
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />
              
              <FormField
                control={form.control}
                name="ay"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Assessment Year *</FormLabel>
                    <Select onValueChange={field.onChange} defaultValue={field.value}>
                      <FormControl>
                        <SelectTrigger data-testid="select-ay">
                          <SelectValue />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="2024-25">2024-25</SelectItem>
                        <SelectItem value="2023-24">2023-24</SelectItem>
                        <SelectItem value="2022-23">2022-23</SelectItem>
                        <SelectItem value="2021-22">2021-22</SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <FormField
                control={form.control}
                name="itrFormType"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>ITR Form Type *</FormLabel>
                    <Select 
                      onValueChange={(value) => {
                        field.onChange(value);
                        handleITRFormChange(value);
                      }} 
                      defaultValue={field.value}
                    >
                      <FormControl>
                        <SelectTrigger data-testid="select-itr-form">
                          <SelectValue />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        {Object.entries(ITR_FORM_TYPES).map(([key, label]) => (
                          <SelectItem key={key} value={key}>{label}</SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />
              
              <FormField
                control={form.control}
                name="tier"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Tier *</FormLabel>
                    <Select onValueChange={field.onChange} defaultValue={field.value}>
                      <FormControl>
                        <SelectTrigger data-testid="select-tier">
                          <SelectValue />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        {Object.entries(TIER_TYPES).map(([key, label]) => (
                          <SelectItem key={key} value={key}>{label}</SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>

            <FormField
              control={form.control}
              name="priority"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Priority</FormLabel>
                  <Select onValueChange={field.onChange} defaultValue={field.value}>
                    <FormControl>
                      <SelectTrigger data-testid="select-priority">
                        <SelectValue />
                      </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                      {Object.entries(PRIORITY_LEVELS).map(([key, label]) => (
                        <SelectItem key={key} value={key}>{label}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <FormMessage />
                </FormItem>
              )}
            />
          </CardContent>
        </Card>

        {/* Due Dates */}
        <Card>
          <CardHeader>
            <CardTitle>Due Dates</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <FormField
                control={form.control}
                name="filingDueDate"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Filing Due Date</FormLabel>
                    <FormControl>
                      <Input 
                        type="date" 
                        {...field}
                        data-testid="input-filing-due"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              
              <FormField
                control={form.control}
                name="auditDueDate"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Audit Due Date</FormLabel>
                    <FormControl>
                      <Input 
                        type="date" 
                        {...field}
                        data-testid="input-audit-due"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>
          </CardContent>
        </Card>

        {/* Additional Options */}
        <Card>
          <CardHeader>
            <CardTitle>Additional Options</CardTitle>
          </CardHeader>
          <CardContent>
            <FormField
              control={form.control}
              name="isAuditApplicable"
              render={({ field }) => (
                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                  <div className="space-y-0.5">
                    <FormLabel className="text-base">Audit Applicable</FormLabel>
                    <div className="text-sm text-muted-foreground">
                      Is this engagement subject to audit requirements (44AB, 44ADA, 44AE)?
                    </div>
                  </div>
                  <FormControl>
                    <Switch
                      checked={field.value}
                      onCheckedChange={field.onChange}
                      data-testid="switch-audit-applicable"
                    />
                  </FormControl>
                </FormItem>
              )}
            />
          </CardContent>
        </Card>

        {/* Submit Button */}
        <div className="flex justify-end space-x-4">
          <Button type="button" variant="outline" data-testid="button-cancel">
            Cancel
          </Button>
          <Button type="submit" disabled={isLoading} data-testid="button-submit">
            {isLoading ? (
              <>
                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                Creating...
              </>
            ) : (
              "Create Engagement"
            )}
          </Button>
        </div>
      </form>
    </Form>
  );
}
