import { Link, useLocation } from "wouter";
import { useAuth } from "@/lib/auth";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { 
  BarChart3, 
  Users, 
  Briefcase, 
  ClipboardCheck, 
  FolderOpen, 
  FileBarChart, 
  Settings, 
  Shield, 
  LogOut,
  FileText
} from "lucide-react";

const navItems = [
  { path: "/dashboard", label: "Dashboard", icon: BarChart3 },
  { path: "/clients", label: "Clients", icon: Users },
  { path: "/engagements", label: "Engagements", icon: Briefcase },
  { path: "/reviews", label: "Reviews", icon: ClipboardCheck, badge: 3 },
  { path: "/documents", label: "Documents", icon: FolderOpen },
  { path: "/reports", label: "Reports", icon: FileBarChart },
];

const bottomNavItems = [
  { path: "/settings", label: "Settings", icon: Settings },
  { path: "/audit", label: "Audit & Policy", icon: Shield },
];

export default function Sidebar() {
  const [location] = useLocation();
  const { user, logout } = useAuth();

  const getUserInitials = (name: string) => {
    return name
      .split(" ")
      .map(n => n[0])
      .join("")
      .toUpperCase()
      .substring(0, 2);
  };

  return (
    <div className="fixed left-0 top-0 h-full w-64 bg-card border-r border-border shadow-sm z-40" data-testid="sidebar">
      {/* Header */}
      <div className="p-6 border-b border-border">
        <div className="flex items-center space-x-3">
          <div className="w-8 h-8 bg-primary rounded-lg flex items-center justify-center">
            <FileText className="w-4 h-4 text-primary-foreground" />
          </div>
          <div>
            <h1 className="font-bold text-lg" data-testid="text-app-title">ITR Tracker</h1>
            <p className="text-xs text-muted-foreground">CA Firm Portal</p>
          </div>
        </div>
      </div>
      
      {/* Navigation */}
      <nav className="p-4 space-y-1">
        {navItems.map((item) => {
          const isActive = location === item.path || (item.path !== "/dashboard" && location.startsWith(item.path));
          const Icon = item.icon;
          
          return (
            <Link key={item.path} href={item.path}>
              <Button
                variant={isActive ? "default" : "ghost"}
                className="w-full justify-start"
                data-testid={`nav-${item.label.toLowerCase().replace(/\s+/g, '-')}`}
              >
                <Icon className="mr-3 h-4 w-4" />
                <span>{item.label}</span>
                {item.badge && (
                  <span className="ml-auto bg-destructive text-destructive-foreground px-2 py-0.5 rounded-full text-xs">
                    {item.badge}
                  </span>
                )}
              </Button>
            </Link>
          );
        })}
        
        <div className="border-t border-border my-4"></div>
        
        {bottomNavItems.map((item) => {
          const isActive = location === item.path;
          const Icon = item.icon;
          
          return (
            <Link key={item.path} href={item.path}>
              <Button
                variant={isActive ? "default" : "ghost"}
                className="w-full justify-start"
                data-testid={`nav-${item.label.toLowerCase().replace(/\s+/g, '-')}`}
              >
                <Icon className="mr-3 h-4 w-4" />
                <span>{item.label}</span>
              </Button>
            </Link>
          );
        })}
      </nav>
      
      {/* User Profile */}
      <div className="absolute bottom-0 left-0 right-0 p-4 border-t border-border">
        <div className="flex items-center space-x-3">
          <Avatar>
            <AvatarFallback className="bg-secondary text-secondary-foreground">
              {user ? getUserInitials(user.name) : "U"}
            </AvatarFallback>
          </Avatar>
          <div className="flex-1 min-w-0">
            <p className="text-sm font-medium truncate" data-testid="text-username">
              {user?.name || "Unknown User"}
            </p>
            <p className="text-xs text-muted-foreground" data-testid="text-user-role">
              {user?.role || "Unknown Role"}
            </p>
          </div>
          <Button 
            variant="ghost" 
            size="icon"
            onClick={logout}
            data-testid="button-logout"
          >
            <LogOut className="h-4 w-4" />
          </Button>
        </div>
      </div>
    </div>
  );
}
