import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import StatusChip from "@/components/common/status-chip";
import { Check, Edit } from "lucide-react";

interface ReviewItem {
  id: string;
  clientName: string;
  details: string;
  stage: string;
  preparer: string;
  reviewer: string;
  due: string;
}

interface ReviewCardProps {
  item: ReviewItem;
}

export default function ReviewCard({ item }: ReviewCardProps) {
  const getStageStatus = (stage: string) => {
    switch (stage) {
      case "Partner Review":
        return "PARTNER_REVIEW";
      case "Manager Review":
        return "MANAGER_REVIEW";
      default:
        return "PENDING";
    }
  };

  const getDueColor = (due: string) => {
    if (due === "Today" || due === "Tomorrow") {
      return "text-red-600 font-medium";
    }
    return "font-medium";
  };

  return (
    <Card className="hover:shadow-md transition-shadow" data-testid={`review-card-${item.id}`}>
      <CardContent className="p-4">
        <div className="flex items-center justify-between mb-3">
          <div>
            <h4 className="font-medium" data-testid="text-client-name">{item.clientName}</h4>
            <p className="text-sm text-muted-foreground" data-testid="text-details">{item.details}</p>
          </div>
          <StatusChip status={getStageStatus(item.stage)} />
        </div>
        
        <div className="space-y-2 mb-4">
          <div className="flex justify-between text-sm">
            <span className="text-muted-foreground">Prepared by:</span>
            <span data-testid="text-preparer">{item.preparer}</span>
          </div>
          <div className="flex justify-between text-sm">
            <span className="text-muted-foreground">Reviewed by:</span>
            <span data-testid="text-reviewer">{item.reviewer}</span>
          </div>
          <div className="flex justify-between text-sm">
            <span className="text-muted-foreground">Due:</span>
            <span className={getDueColor(item.due)} data-testid="text-due">{item.due}</span>
          </div>
        </div>
        
        <div className="flex space-x-2">
          <Button 
            className="flex-1 bg-green-500 text-white hover:bg-green-600" 
            size="sm"
            data-testid="button-approve"
          >
            <Check className="mr-1 h-4 w-4" />
            Approve
          </Button>
          <Button 
            className="flex-1 bg-orange-500 text-white hover:bg-orange-600" 
            size="sm"
            data-testid="button-request-changes"
          >
            <Edit className="mr-1 h-4 w-4" />
            Request Changes
          </Button>
        </div>
      </CardContent>
    </Card>
  );
}
