import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Shield, Eye, Download } from "lucide-react";
import type { ActivityLogEntry } from "@shared/schema";

interface ActivityLogTableProps {
  data: ActivityLogEntry[];
  total: number;
  isLoading: boolean;
  filters: {
    q?: string;
    entityType?: string;
    action?: string;
    limit?: number;
    offset?: number;
  };
  onFiltersChange: (filters: any) => void;
}

export default function ActivityLogTable({ 
  data, 
  total, 
  isLoading, 
  filters, 
  onFiltersChange 
}: ActivityLogTableProps) {
  
  const getActionColor = (action: string) => {
    const colorMap: Record<string, string> = {
      "CREATE": "bg-green-100 text-green-800",
      "UPDATE": "bg-blue-100 text-blue-800", 
      "DELETE": "bg-red-100 text-red-800",
      "LOGIN": "bg-purple-100 text-purple-800",
      "LOGOUT": "bg-gray-100 text-gray-800",
      "STATUS_CHANGE": "bg-orange-100 text-orange-800",
      "APPROVE_REVIEW": "bg-emerald-100 text-emerald-800",
      "REJECT_REVIEW": "bg-red-100 text-red-800",
      "UPLOAD_DOCUMENT": "bg-blue-100 text-blue-800",
      "DOWNLOAD_DOCUMENT": "bg-indigo-100 text-indigo-800",
      "ADD_COMMENT": "bg-purple-100 text-purple-800",
      "SLA_BREACH": "bg-red-100 text-red-800",
      "EXPORT_DATA": "bg-yellow-100 text-yellow-800"
    };
    return colorMap[action] || "bg-gray-100 text-gray-800";
  };

  const getRoleColor = (role: string) => {
    const colorMap: Record<string, string> = {
      "PARTNER": "bg-purple-100 text-purple-800",
      "MANAGER": "bg-orange-100 text-orange-800",
      "STAFF": "bg-blue-100 text-blue-800",
      "CLIENT": "bg-green-100 text-green-800",
      "AUDITOR": "bg-gray-100 text-gray-800",
      "SYSTEM": "bg-slate-100 text-slate-800"
    };
    return colorMap[role] || "bg-gray-100 text-gray-800";
  };

  const formatTimestamp = (timestamp: string | Date) => {
    const date = new Date(timestamp);
    return {
      date: date.toLocaleDateString(),
      time: date.toLocaleTimeString()
    };
  };

  const maskIP = (ip: string) => {
    // Mask IP address for privacy (show only first two octets)
    const parts = ip.split('.');
    if (parts.length === 4) {
      return `${parts[0]}.${parts[1]}.***.**`;
    }
    return ip;
  };

  const handleVerifySignature = (logEntry: ActivityLogEntry) => {
    // In a real implementation, this would call an API to verify the HMAC signature
    console.log("Verifying signature for log entry:", logEntry.id);
    // Show verification result
  };

  if (isLoading) {
    return (
      <Card>
        <CardHeader>
          <Skeleton className="h-6 w-32" />
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {Array.from({ length: 10 }).map((_, i) => (
              <Skeleton key={i} className="h-16 w-full" />
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card data-testid="activity-log-table">
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle className="flex items-center space-x-2">
            <Shield className="h-5 w-5" />
            <span>Activity Log ({total})</span>
          </CardTitle>
          <div className="flex items-center space-x-2">
            <div className="flex items-center space-x-2">
              <Shield className="h-4 w-4 text-green-600" />
              <span className="text-sm text-green-600 font-medium" data-testid="text-integrity-status">
                Integrity Verified
              </span>
            </div>
          </div>
        </div>
      </CardHeader>
      
      <CardContent>
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Timestamp</TableHead>
                <TableHead>User</TableHead>
                <TableHead>Action</TableHead>
                <TableHead>Entity</TableHead>
                <TableHead>IP Address</TableHead>
                <TableHead>Signature</TableHead>
                <TableHead className="text-right">Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {data.map((log) => {
                const timestamp = formatTimestamp(log.createdAt || new Date());
                
                return (
                  <TableRow key={log.id} className="hover:bg-muted/50" data-testid={`log-entry-${log.id}`}>
                    <TableCell className="font-mono">
                      <div>
                        <p className="text-sm" data-testid="text-log-date">{timestamp.date}</p>
                        <p className="text-xs text-muted-foreground" data-testid="text-log-time">{timestamp.time}</p>
                      </div>
                    </TableCell>
                    
                    <TableCell>
                      <div>
                        <p className="text-sm font-medium" data-testid="text-actor-id">
                          {log.actorUserId || "System"}
                        </p>
                        <Badge className={`text-xs ${getRoleColor(log.actorRole)}`} data-testid="badge-actor-role">
                          {log.actorRole}
                        </Badge>
                      </div>
                    </TableCell>
                    
                    <TableCell>
                      <Badge className={getActionColor(log.action)} data-testid="badge-action">
                        {log.action.replace(/_/g, ' ')}
                      </Badge>
                    </TableCell>
                    
                    <TableCell>
                      <div>
                        <p className="text-sm" data-testid="text-entity-type">{log.entityType}</p>
                        <p className="text-xs text-muted-foreground font-mono" data-testid="text-entity-id">
                          {log.entityId.substring(0, 8)}...
                        </p>
                      </div>
                    </TableCell>
                    
                    <TableCell className="font-mono text-sm" data-testid="text-ip-address">
                      {maskIP(log.ip)}
                    </TableCell>
                    
                    <TableCell>
                      <Button 
                        variant="ghost" 
                        size="icon"
                        onClick={() => handleVerifySignature(log)}
                        title="Verify HMAC signature"
                        data-testid="button-verify-signature"
                      >
                        <Shield className="h-4 w-4 text-green-600" />
                      </Button>
                    </TableCell>
                    
                    <TableCell className="text-right">
                      <Button 
                        variant="ghost" 
                        size="icon"
                        title="View details"
                        data-testid="button-view-details"
                      >
                        <Eye className="h-4 w-4" />
                      </Button>
                    </TableCell>
                  </TableRow>
                );
              })}
            </TableBody>
          </Table>
        </div>

        {/* Pagination */}
        <div className="flex items-center justify-between pt-4">
          <p className="text-sm text-muted-foreground" data-testid="text-pagination-info">
            Showing {(filters.offset || 0) + 1}-{Math.min((filters.offset || 0) + (filters.limit || 50), total)} of {total} entries
          </p>
          <div className="flex items-center space-x-2">
            <Button 
              variant="outline" 
              size="sm" 
              disabled={!filters.offset || filters.offset === 0}
              onClick={() => onFiltersChange({...filters, offset: Math.max(0, (filters.offset || 0) - (filters.limit || 50))})}
              data-testid="button-previous-page"
            >
              Previous
            </Button>
            <Button 
              variant="outline" 
              size="sm"
              disabled={(filters.offset || 0) + (filters.limit || 50) >= total}
              onClick={() => onFiltersChange({...filters, offset: (filters.offset || 0) + (filters.limit || 50)})}
              data-testid="button-next-page"
            >
              Next
            </Button>
          </div>
        </div>

        {data.length === 0 && (
          <div className="text-center py-8">
            <Shield className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
            <p className="text-lg font-medium mb-2">No activity logs found</p>
            <p className="text-muted-foreground">No matching activity logs for the selected filters</p>
          </div>
        )}
      </CardContent>
    </Card>
  );
}
