import { useState } from "react";
import { useEngagements } from "@/hooks/use-engagements";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Skeleton } from "@/components/ui/skeleton";
import StatusChip from "@/components/common/status-chip";
import PriorityBadge from "@/components/common/priority-badge";
import { Eye, MoreHorizontal, Download } from "lucide-react";
import { Link } from "wouter";

export default function EngagementsTable() {
  const [filters, setFilters] = useState({
    status: "",
    limit: 10,
    offset: 0
  });

  const { data, isLoading } = useEngagements(filters);

  const maskPAN = (pan: string) => {
    if (!pan || pan.length !== 10) return pan;
    return pan.substring(0, 2) + "***" + pan.substring(5);
  };

  const formatDueDate = (dueDate: string | Date) => {
    const date = new Date(dueDate);
    const now = new Date();
    const diffInDays = Math.ceil((date.getTime() - now.getTime()) / (1000 * 60 * 60 * 24));
    
    if (diffInDays < 0) {
      return {
        text: date.toLocaleDateString(),
        subtext: `${Math.abs(diffInDays)} days overdue`,
        color: "text-red-600"
      };
    } else if (diffInDays <= 5) {
      return {
        text: date.toLocaleDateString(),
        subtext: `${diffInDays} days remaining`,
        color: "text-orange-600"
      };
    } else {
      return {
        text: date.toLocaleDateString(),
        subtext: `${diffInDays} days remaining`,
        color: "text-green-600"
      };
    }
  };

  if (isLoading) {
    return (
      <Card>
        <CardHeader>
          <Skeleton className="h-6 w-48" />
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {Array.from({ length: 5 }).map((_, i) => (
              <Skeleton key={i} className="h-16 w-full" />
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card data-testid="engagements-table">
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Active Engagements</CardTitle>
          <div className="flex items-center space-x-3">
            <div className="flex items-center space-x-2">
              <label className="text-sm text-muted-foreground">Filter:</label>
              <Select value={filters.status} onValueChange={(value) => setFilters({...filters, status: value})}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="All Statuses" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="">All Statuses</SelectItem>
                  <SelectItem value="MANAGER_REVIEW">Pending Review</SelectItem>
                  <SelectItem value="PARTNER_REVIEW">Partner Review</SelectItem>
                  <SelectItem value="ITR_FILED_ACK_CAPTURED">Filed</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <Button variant="outline" size="sm" data-testid="button-export">
              <Download className="mr-1 h-4 w-4" />
              Export CSV
            </Button>
          </div>
        </div>
      </CardHeader>
      
      <CardContent>
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Client</TableHead>
                <TableHead>AY</TableHead>
                <TableHead>Form</TableHead>
                <TableHead>Status</TableHead>
                <TableHead>Assignee</TableHead>
                <TableHead>Due Date</TableHead>
                <TableHead>Priority</TableHead>
                <TableHead className="text-right">Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {data?.data.map((engagement) => {
                const dueDate = engagement.dueDates && (engagement.dueDates as any).filingDue 
                  ? formatDueDate((engagement.dueDates as any).filingDue)
                  : null;

                return (
                  <TableRow key={engagement.id} className="hover:bg-muted/50" data-testid={`engagement-row-${engagement.id}`}>
                    <TableCell>
                      <div>
                        <p className="text-sm font-medium" data-testid="text-client-name">
                          {engagement.client.displayName}
                        </p>
                        <p className="text-xs text-muted-foreground" data-testid="text-client-pan">
                          PAN: {maskPAN(engagement.client.pan)}
                        </p>
                      </div>
                    </TableCell>
                    <TableCell className="text-sm" data-testid="text-ay">{engagement.ay}</TableCell>
                    <TableCell className="text-sm font-medium" data-testid="text-form-type">{engagement.itrFormType}</TableCell>
                    <TableCell>
                      <StatusChip status={engagement.status} />
                    </TableCell>
                    <TableCell className="text-sm" data-testid="text-assignee">
                      {/* This would come from assignments table */}
                      Staff Member
                    </TableCell>
                    <TableCell>
                      {dueDate ? (
                        <div>
                          <span className={`text-sm font-medium ${dueDate.color}`} data-testid="text-due-date">
                            {dueDate.text}
                          </span>
                          <p className="text-xs text-muted-foreground" data-testid="text-due-subtext">
                            {dueDate.subtext}
                          </p>
                        </div>
                      ) : (
                        <span className="text-sm text-muted-foreground">No due date</span>
                      )}
                    </TableCell>
                    <TableCell>
                      <PriorityBadge priority={engagement.priority} />
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex justify-end space-x-2">
                        <Link href={`/engagements/${engagement.id}`}>
                          <Button variant="ghost" size="icon" data-testid="button-view-engagement">
                            <Eye className="h-4 w-4" />
                          </Button>
                        </Link>
                        <Button variant="ghost" size="icon" data-testid="button-engagement-menu">
                          <MoreHorizontal className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                );
              })}
            </TableBody>
          </Table>
        </div>
        
        {/* Pagination */}
        <div className="flex items-center justify-between pt-4">
          <p className="text-sm text-muted-foreground" data-testid="text-pagination-info">
            Showing {data?.offset || 0 + 1}-{Math.min((data?.offset || 0) + (data?.limit || 0), data?.total || 0)} of {data?.total || 0} engagements
          </p>
          <div className="flex items-center space-x-2">
            <Button 
              variant="outline" 
              size="sm" 
              disabled={!data?.offset || data.offset === 0}
              onClick={() => setFilters({...filters, offset: Math.max(0, (data?.offset || 0) - (data?.limit || 10))})}
              data-testid="button-previous-page"
            >
              Previous
            </Button>
            <Button 
              variant="outline" 
              size="sm"
              disabled={!data || (data.offset + data.limit) >= data.total}
              onClick={() => setFilters({...filters, offset: (data?.offset || 0) + (data?.limit || 10)})}
              data-testid="button-next-page"
            >
              Next
            </Button>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}
