import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import type { Client, InsertClient } from "@shared/schema";
import type { SearchFilters, PaginatedResponse } from "@/lib/types";

export function useClients(filters?: SearchFilters) {
  return useQuery<PaginatedResponse<Client>>({
    queryKey: ["/api/clients", filters],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (filters?.q) params.append("q", filters.q);
      if (filters?.limit) params.append("limit", filters.limit.toString());
      if (filters?.offset) params.append("offset", filters.offset.toString());
      
      const response = await fetch(`/api/clients?${params}`, {
        credentials: "include",
        headers: {
          Authorization: `Bearer ${localStorage.getItem("accessToken")}`,
        },
      });
      
      if (!response.ok) throw new Error("Failed to fetch clients");
      const result = await response.json();
      
      return {
        data: result.clients,
        total: result.total,
        limit: filters?.limit || 50,
        offset: filters?.offset || 0,
      };
    },
  });
}

export function useClient(id: string) {
  return useQuery<Client>({
    queryKey: ["/api/clients", id],
    queryFn: async () => {
      const response = await fetch(`/api/clients/${id}`, {
        credentials: "include",
        headers: {
          Authorization: `Bearer ${localStorage.getItem("accessToken")}`,
        },
      });
      
      if (!response.ok) throw new Error("Failed to fetch client");
      return response.json();
    },
    enabled: !!id,
  });
}

export function useCreateClient() {
  const queryClient = useQueryClient();
  
  return useMutation<Client, Error, InsertClient>({
    mutationFn: async (clientData) => {
      const response = await apiRequest("POST", "/api/clients", clientData);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/clients"] });
    },
  });
}

export function useUpdateClient() {
  const queryClient = useQueryClient();
  
  return useMutation<Client, Error, { id: string; data: Partial<Client> }>({
    mutationFn: async ({ id, data }) => {
      const response = await apiRequest("PUT", `/api/clients/${id}`, data);
      return response.json();
    },
    onSuccess: (_, { id }) => {
      queryClient.invalidateQueries({ queryKey: ["/api/clients"] });
      queryClient.invalidateQueries({ queryKey: ["/api/clients", id] });
    },
  });
}
