import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import type { Engagement, InsertEngagement, ChecklistItem, Comment, Document } from "@shared/schema";
import type { SearchFilters, PaginatedResponse, EngagementWithClient, CommentWithAuthor, StatusTransition } from "@/lib/types";

export function useEngagements(filters?: SearchFilters) {
  return useQuery<PaginatedResponse<EngagementWithClient>>({
    queryKey: ["/api/engagements", filters],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (filters?.q) params.append("q", filters.q);
      if (filters?.limit) params.append("limit", filters.limit.toString());
      if (filters?.offset) params.append("offset", filters.offset.toString());
      if (filters?.status) params.append("status", filters.status);
      if (filters?.priority) params.append("priority", filters.priority);
      if (filters?.assignee) params.append("assignee", filters.assignee);
      if (filters?.ay) params.append("ay", filters.ay);
      
      const response = await fetch(`/api/engagements?${params}`, {
        credentials: "include",
        headers: {
          Authorization: `Bearer ${localStorage.getItem("accessToken")}`,
        },
      });
      
      if (!response.ok) throw new Error("Failed to fetch engagements");
      const result = await response.json();
      
      return {
        data: result.engagements,
        total: result.total,
        limit: filters?.limit || 50,
        offset: filters?.offset || 0,
      };
    },
  });
}

export function useEngagement(id: string) {
  return useQuery<Engagement>({
    queryKey: ["/api/engagements", id],
    queryFn: async () => {
      const response = await fetch(`/api/engagements/${id}`, {
        credentials: "include",
        headers: {
          Authorization: `Bearer ${localStorage.getItem("accessToken")}`,
        },
      });
      
      if (!response.ok) throw new Error("Failed to fetch engagement");
      return response.json();
    },
    enabled: !!id,
  });
}

export function useCreateEngagement() {
  const queryClient = useQueryClient();
  
  return useMutation<Engagement, Error, InsertEngagement>({
    mutationFn: async (engagementData) => {
      const response = await apiRequest("POST", "/api/engagements", engagementData);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/engagements"] });
    },
  });
}

export function useTransitionStatus() {
  const queryClient = useQueryClient();
  
  return useMutation<Engagement, Error, { id: string; transition: StatusTransition }>({
    mutationFn: async ({ id, transition }) => {
      const response = await apiRequest("POST", `/api/engagements/${id}/transition`, transition);
      return response.json();
    },
    onSuccess: (_, { id }) => {
      queryClient.invalidateQueries({ queryKey: ["/api/engagements"] });
      queryClient.invalidateQueries({ queryKey: ["/api/engagements", id] });
    },
  });
}

export function useEngagementDocuments(engagementId: string) {
  return useQuery<Document[]>({
    queryKey: ["/api/engagements", engagementId, "documents"],
    queryFn: async () => {
      const response = await fetch(`/api/engagements/${engagementId}/documents`, {
        credentials: "include",
        headers: {
          Authorization: `Bearer ${localStorage.getItem("accessToken")}`,
        },
      });
      
      if (!response.ok) throw new Error("Failed to fetch documents");
      return response.json();
    },
    enabled: !!engagementId,
  });
}

export function useEngagementComments(engagementId: string) {
  return useQuery<CommentWithAuthor[]>({
    queryKey: ["/api/engagements", engagementId, "comments"],
    queryFn: async () => {
      const response = await fetch(`/api/engagements/${engagementId}/comments`, {
        credentials: "include",
        headers: {
          Authorization: `Bearer ${localStorage.getItem("accessToken")}`,
        },
      });
      
      if (!response.ok) throw new Error("Failed to fetch comments");
      return response.json();
    },
    enabled: !!engagementId,
  });
}

export function useEngagementChecklist(engagementId: string) {
  return useQuery<ChecklistItem[]>({
    queryKey: ["/api/engagements", engagementId, "checklist"],
    queryFn: async () => {
      const response = await fetch(`/api/engagements/${engagementId}/checklist`, {
        credentials: "include",
        headers: {
          Authorization: `Bearer ${localStorage.getItem("accessToken")}`,
        },
      });
      
      if (!response.ok) throw new Error("Failed to fetch checklist");
      return response.json();
    },
    enabled: !!engagementId,
  });
}

export function useUpdateChecklistItem() {
  const queryClient = useQueryClient();
  
  return useMutation<ChecklistItem, Error, { id: string; data: Partial<ChecklistItem> }>({
    mutationFn: async ({ id, data }) => {
      const response = await apiRequest("PUT", `/api/checklist-items/${id}`, data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/engagements"] });
    },
  });
}

export function useCreateComment() {
  const queryClient = useQueryClient();
  
  return useMutation<Comment, Error, { engagementId: string; entityType: string; entityId: string; body: string; mentions?: string[] }>({
    mutationFn: async (commentData) => {
      const response = await apiRequest("POST", "/api/comments", commentData);
      return response.json();
    },
    onSuccess: (_, { engagementId }) => {
      queryClient.invalidateQueries({ queryKey: ["/api/engagements", engagementId, "comments"] });
    },
  });
}
