import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import ActivityLogTable from "@/components/tables/activity-log-table";
import { Shield, Download, Search, FileText, Database, Eye, Filter } from "lucide-react";
import type { ActivityLogEntry } from "@shared/schema";

export default function Audit() {
  const [search, setSearch] = useState("");
  const [filters, setFilters] = useState({
    q: "",
    entityType: "",
    action: "",
    limit: 50,
    offset: 0
  });

  const { data: activityLog, isLoading } = useQuery<{logs: ActivityLogEntry[], total: number}>({
    queryKey: ["/api/activity-log", filters],
  });

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    setFilters({ ...filters, q: search, offset: 0 });
  };

  const handleExportLog = async (format: 'csv' | 'pdf') => {
    try {
      const response = await fetch(`/api/activity-log/export?format=${format}`, {
        headers: {
          Authorization: `Bearer ${localStorage.getItem("accessToken")}`,
        },
      });
      
      if (response.ok) {
        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `activity-log.${format}`;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
      }
    } catch (error) {
      console.error(`Failed to export ${format}:`, error);
    }
  };

  return (
    <div className="p-6 space-y-6" data-testid="audit-page">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold flex items-center space-x-3" data-testid="text-page-title">
            <Shield className="h-6 w-6" />
            <span>Audit & Policy</span>
          </h1>
          <p className="text-muted-foreground">Compliance monitoring and audit trails</p>
        </div>
        
        <div className="flex items-center space-x-3">
          <div className="flex items-center space-x-2">
            <Shield className="h-4 w-4 text-green-600" />
            <span className="text-sm text-green-600 font-medium" data-testid="text-integrity-status">
              Integrity Verified
            </span>
          </div>
          <Button variant="outline" onClick={() => handleExportLog('csv')} data-testid="button-export-csv">
            <Download className="mr-2 h-4 w-4" />
            Export CSV
          </Button>
          <Button variant="outline" onClick={() => handleExportLog('pdf')} data-testid="button-export-pdf">
            <Download className="mr-2 h-4 w-4" />
            Export PDF
          </Button>
        </div>
      </div>

      {/* Compliance Overview */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                <Shield className="h-6 w-6 text-green-600" />
              </div>
              <div>
                <p className="text-2xl font-bold text-green-600" data-testid="text-log-entries">
                  {activityLog?.total || 0}
                </p>
                <p className="text-sm text-muted-foreground">Log Entries</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                <Database className="h-6 w-6 text-blue-600" />
              </div>
              <div>
                <p className="text-2xl font-bold" data-testid="text-integrity-checks">100%</p>
                <p className="text-sm text-muted-foreground">Integrity Verified</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center">
                <FileText className="h-6 w-6 text-purple-600" />
              </div>
              <div>
                <p className="text-2xl font-bold" data-testid="text-compliance-score">98.5%</p>
                <p className="text-sm text-muted-foreground">Compliance Score</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Policy Status */}
      <Card>
        <CardHeader>
          <CardTitle>Policy & Compliance Status</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-4">
              <h4 className="font-medium">Data Retention Policy</h4>
              <div className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Client Records:</span>
                  <span className="font-medium" data-testid="text-client-retention">8 years</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Activity Logs:</span>
                  <span className="font-medium" data-testid="text-logs-retention">10 years</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Documents:</span>
                  <span className="font-medium" data-testid="text-documents-retention">8 years</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Computation Records:</span>
                  <span className="font-medium" data-testid="text-computation-retention">8 years</span>
                </div>
              </div>
            </div>

            <div className="space-y-4">
              <h4 className="font-medium">Compliance Status</h4>
              <div className="space-y-3">
                <div className="flex items-center justify-between">
                  <span className="text-sm">DPDP Act 2023</span>
                  <Badge className="bg-green-100 text-green-800" data-testid="badge-dpdp-compliance">
                    ✓ Compliant
                  </Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm">IT Act 2000 (Sec 65B)</span>
                  <Badge className="bg-green-100 text-green-800" data-testid="badge-it-act-compliance">
                    ✓ Compliant
                  </Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm">ICAI Code of Ethics</span>
                  <Badge className="bg-green-100 text-green-800" data-testid="badge-icai-compliance">
                    ✓ Compliant
                  </Badge>
                </div>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Search and Filters */}
      <Card>
        <CardContent className="p-6">
          <form onSubmit={handleSearch} className="space-y-4">
            <div className="flex items-center space-x-4">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
                <Input
                  type="search"
                  placeholder="Search activity logs..."
                  value={search}
                  onChange={(e) => setSearch(e.target.value)}
                  className="pl-10"
                  data-testid="input-search"
                />
              </div>
              <Button type="submit" data-testid="button-search">Search</Button>
            </div>
            
            <div className="flex items-center space-x-4">
              <Filter className="h-4 w-4 text-muted-foreground" />
              <Select value={filters.entityType} onValueChange={(value) => setFilters({...filters, entityType: value, offset: 0})}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Entity Type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="">All Types</SelectItem>
                  <SelectItem value="USER">User</SelectItem>
                  <SelectItem value="CLIENT">Client</SelectItem>
                  <SelectItem value="ENGAGEMENT">Engagement</SelectItem>
                  <SelectItem value="DOCUMENT">Document</SelectItem>
                  <SelectItem value="COMMENT">Comment</SelectItem>
                </SelectContent>
              </Select>
              
              <Select value={filters.action} onValueChange={(value) => setFilters({...filters, action: value, offset: 0})}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Action Type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="">All Actions</SelectItem>
                  <SelectItem value="CREATE">Create</SelectItem>
                  <SelectItem value="UPDATE">Update</SelectItem>
                  <SelectItem value="DELETE">Delete</SelectItem>
                  <SelectItem value="LOGIN">Login</SelectItem>
                  <SelectItem value="STATUS_CHANGE">Status Change</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </form>
        </CardContent>
      </Card>

      {/* Activity Log Table */}
      <ActivityLogTable 
        data={activityLog?.logs || []} 
        total={activityLog?.total || 0}
        isLoading={isLoading}
        filters={filters}
        onFiltersChange={setFilters}
      />

      {/* Compliance Actions */}
      <Card>
        <CardHeader>
          <CardTitle>Compliance Actions</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            <Button variant="outline" className="h-20 flex flex-col space-y-2" data-testid="button-compliance-report">
              <FileText className="h-6 w-6" />
              <span>Compliance Report</span>
            </Button>
            <Button variant="outline" className="h-20 flex flex-col space-y-2" data-testid="button-audit-trail">
              <Shield className="h-6 w-6" />
              <span>Export Audit Trail</span>
            </Button>
            <Button variant="outline" className="h-20 flex flex-col space-y-2" data-testid="button-review-consents">
              <Eye className="h-6 w-6" />
              <span>Review Consents</span>
            </Button>
            <Button variant="outline" className="h-20 flex flex-col space-y-2" data-testid="button-retention-review">
              <Database className="h-6 w-6" />
              <span>Retention Review</span>
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
