import { useState } from "react";
import { useClients, useCreateClient } from "@/hooks/use-clients";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import ClientForm from "@/components/forms/client-form";
import { Search, Plus, Eye, Edit, Download } from "lucide-react";
import { Link } from "wouter";
import type { InsertClient } from "@shared/schema";

export default function Clients() {
  const [search, setSearch] = useState("");
  const [isCreateDialogOpen, setIsCreateDialogOpen] = useState(false);
  const [filters, setFilters] = useState({
    q: "",
    limit: 50,
    offset: 0
  });

  const { data, isLoading } = useClients(filters);
  const createClient = useCreateClient();

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    setFilters({ ...filters, q: search, offset: 0 });
  };

  const handleCreateClient = async (clientData: InsertClient) => {
    try {
      await createClient.mutateAsync(clientData);
      setIsCreateDialogOpen(false);
    } catch (error) {
      console.error("Failed to create client:", error);
    }
  };

  const maskPAN = (pan: string) => {
    if (!pan || pan.length !== 10) return pan;
    return pan.substring(0, 2) + "***" + pan.substring(5);
  };

  if (isLoading) {
    return (
      <div className="p-6 space-y-6">
        <Skeleton className="h-8 w-48" />
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-32" />
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {Array.from({ length: 10 }).map((_, i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6" data-testid="clients-page">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">Clients</h1>
          <p className="text-muted-foreground">Manage your client database</p>
        </div>
        
        <Dialog open={isCreateDialogOpen} onOpenChange={setIsCreateDialogOpen}>
          <DialogTrigger asChild>
            <Button data-testid="button-new-client">
              <Plus className="mr-2 h-4 w-4" />
              New Client
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>Create New Client</DialogTitle>
            </DialogHeader>
            <ClientForm 
              onSubmit={handleCreateClient}
              isLoading={createClient.isPending}
            />
          </DialogContent>
        </Dialog>
      </div>

      {/* Search and Filters */}
      <Card>
        <CardContent className="p-6">
          <form onSubmit={handleSearch} className="flex items-center space-x-4">
            <div className="relative flex-1">
              <Search className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
              <Input
                type="search"
                placeholder="Search clients by name, PAN..."
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                className="pl-10"
                data-testid="input-search"
              />
            </div>
            <Button type="submit" data-testid="button-search">Search</Button>
            <Button variant="outline" data-testid="button-export">
              <Download className="mr-2 h-4 w-4" />
              Export
            </Button>
          </form>
        </CardContent>
      </Card>

      {/* Clients Table */}
      <Card>
        <CardHeader>
          <CardTitle>Clients ({data?.total || 0})</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Name</TableHead>
                  <TableHead>PAN</TableHead>
                  <TableHead>Legal Name</TableHead>
                  <TableHead>Risk Category</TableHead>
                  <TableHead>Created</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {data?.data.map((client) => (
                  <TableRow key={client.id} className="hover:bg-muted/50" data-testid={`client-row-${client.id}`}>
                    <TableCell>
                      <div>
                        <p className="font-medium" data-testid="text-display-name">{client.displayName}</p>
                        {client.contacts && (client.contacts as any[]).length > 0 && (
                          <p className="text-xs text-muted-foreground">
                            {(client.contacts as any)[0].email}
                          </p>
                        )}
                      </div>
                    </TableCell>
                    <TableCell className="font-mono text-sm" data-testid="text-pan">
                      {maskPAN(client.pan)}
                    </TableCell>
                    <TableCell className="text-sm" data-testid="text-legal-name">
                      {client.legalName}
                    </TableCell>
                    <TableCell>
                      <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                        client.riskCategory === "HIGH" ? "bg-red-100 text-red-800" :
                        client.riskCategory === "MEDIUM" ? "bg-yellow-100 text-yellow-800" :
                        "bg-green-100 text-green-800"
                      }`} data-testid="badge-risk-category">
                        {client.riskCategory}
                      </span>
                    </TableCell>
                    <TableCell className="text-sm text-muted-foreground" data-testid="text-created-date">
                      {client.createdAt ? new Date(client.createdAt).toLocaleDateString() : "N/A"}
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex justify-end space-x-2">
                        <Link href={`/clients/${client.id}`}>
                          <Button variant="ghost" size="icon" data-testid="button-view-client">
                            <Eye className="h-4 w-4" />
                          </Button>
                        </Link>
                        <Button variant="ghost" size="icon" data-testid="button-edit-client">
                          <Edit className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>

          {/* Pagination */}
          <div className="flex items-center justify-between pt-4">
            <p className="text-sm text-muted-foreground" data-testid="text-pagination-info">
              Showing {(filters.offset || 0) + 1}-{Math.min((filters.offset || 0) + (filters.limit || 50), data?.total || 0)} of {data?.total || 0} clients
            </p>
            <div className="flex items-center space-x-2">
              <Button 
                variant="outline" 
                size="sm" 
                disabled={!filters.offset || filters.offset === 0}
                onClick={() => setFilters({...filters, offset: Math.max(0, (filters.offset || 0) - (filters.limit || 50))})}
                data-testid="button-previous-page"
              >
                Previous
              </Button>
              <Button 
                variant="outline" 
                size="sm"
                disabled={!data || (filters.offset + filters.limit) >= data.total}
                onClick={() => setFilters({...filters, offset: (filters.offset || 0) + (filters.limit || 50)})}
                data-testid="button-next-page"
              >
                Next
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
