import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Search, Download, Eye, Trash2, Upload, FileText, FileX, Filter } from "lucide-react";
import { DOCUMENT_TYPES } from "@/lib/constants";

interface DocumentWithEngagement {
  id: string;
  fileName: string;
  docType: string;
  fileSize: number;
  createdAt: string;
  uploadedBy: string;
  clientVisible: boolean;
  version: number;
  engagement: {
    id: string;
    client: {
      displayName: string;
      pan: string;
    };
    ay: string;
    itrFormType: string;
  };
}

export default function Documents() {
  const [search, setSearch] = useState("");
  const [filters, setFilters] = useState({
    q: "",
    docType: "",
    engagementId: "",
    limit: 50,
    offset: 0
  });

  // Mock query for now - would be replaced with actual API call
  const { data, isLoading } = useQuery<{documents: DocumentWithEngagement[], total: number}>({
    queryKey: ["/api/documents", filters],
    queryFn: async () => {
      // This would be a real API call to /api/documents with proper filtering
      return {
        documents: [],
        total: 0
      };
    }
  });

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    setFilters({ ...filters, q: search, offset: 0 });
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return "0 Bytes";
    const k = 1024;
    const sizes = ["Bytes", "KB", "MB", "GB"];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
  };

  const getFileIcon = (docType: string) => {
    switch (docType) {
      case "FORM_16":
      case "AIS_26AS":
      case "TIS":
        return "📄";
      case "BANK_STATEMENT":
        return "🏦";
      case "CAPITAL_GAINS":
        return "📈";
      case "AUDIT_REPORT":
        return "📋";
      default:
        return "📄";
    }
  };

  const maskPAN = (pan: string) => {
    if (!pan || pan.length !== 10) return pan;
    return pan.substring(0, 2) + "***" + pan.substring(5);
  };

  if (isLoading) {
    return (
      <div className="p-6 space-y-6">
        <Skeleton className="h-8 w-48" />
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-32" />
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {Array.from({ length: 10 }).map((_, i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6" data-testid="documents-page">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">Documents</h1>
          <p className="text-muted-foreground">Manage engagement documents and files</p>
        </div>
        
        <Button data-testid="button-upload">
          <Upload className="mr-2 h-4 w-4" />
          Upload Document
        </Button>
      </div>

      {/* Search and Filters */}
      <Card>
        <CardContent className="p-6">
          <form onSubmit={handleSearch} className="space-y-4">
            <div className="flex items-center space-x-4">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
                <Input
                  type="search"
                  placeholder="Search documents by name, client, PAN..."
                  value={search}
                  onChange={(e) => setSearch(e.target.value)}
                  className="pl-10"
                  data-testid="input-search"
                />
              </div>
              <Button type="submit" data-testid="button-search">Search</Button>
            </div>
            
            <div className="flex items-center space-x-4">
              <Filter className="h-4 w-4 text-muted-foreground" />
              <Select value={filters.docType} onValueChange={(value) => setFilters({...filters, docType: value, offset: 0})}>
                <SelectTrigger className="w-48">
                  <SelectValue placeholder="Document Type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="">All Types</SelectItem>
                  {Object.entries(DOCUMENT_TYPES).map(([key, label]) => (
                    <SelectItem key={key} value={key}>{label}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
              
              <Button variant="outline" data-testid="button-export">
                <Download className="mr-2 h-4 w-4" />
                Export List
              </Button>
            </div>
          </form>
        </CardContent>
      </Card>

      {/* Documents Table */}
      <Card>
        <CardHeader>
          <CardTitle>Documents ({data?.total || 0})</CardTitle>
        </CardHeader>
        <CardContent>
          {data?.documents && data.documents.length > 0 ? (
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Document</TableHead>
                    <TableHead>Type</TableHead>
                    <TableHead>Client</TableHead>
                    <TableHead>Engagement</TableHead>
                    <TableHead>Size</TableHead>
                    <TableHead>Uploaded</TableHead>
                    <TableHead>Visibility</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {data.documents.map((document) => (
                    <TableRow key={document.id} className="hover:bg-muted/50" data-testid={`document-row-${document.id}`}>
                      <TableCell>
                        <div className="flex items-center space-x-3">
                          <span className="text-2xl">{getFileIcon(document.docType)}</span>
                          <div>
                            <p className="font-medium" data-testid="text-filename">{document.fileName}</p>
                            <p className="text-xs text-muted-foreground">v{document.version}</p>
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline" data-testid="badge-doc-type">
                          {DOCUMENT_TYPES[document.docType as keyof typeof DOCUMENT_TYPES] || document.docType}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div>
                          <p className="text-sm font-medium" data-testid="text-client-name">
                            {document.engagement.client.displayName}
                          </p>
                          <p className="text-xs text-muted-foreground" data-testid="text-client-pan">
                            {maskPAN(document.engagement.client.pan)}
                          </p>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div>
                          <p className="text-sm" data-testid="text-engagement-form">{document.engagement.itrFormType}</p>
                          <p className="text-xs text-muted-foreground" data-testid="text-engagement-ay">{document.engagement.ay}</p>
                        </div>
                      </TableCell>
                      <TableCell className="text-sm" data-testid="text-file-size">
                        {formatFileSize(document.fileSize)}
                      </TableCell>
                      <TableCell className="text-sm text-muted-foreground" data-testid="text-upload-date">
                        {new Date(document.createdAt).toLocaleDateString()}
                      </TableCell>
                      <TableCell>
                        <Badge variant={document.clientVisible ? "default" : "secondary"} data-testid="badge-visibility">
                          {document.clientVisible ? "Client Visible" : "Internal"}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-right">
                        <div className="flex justify-end space-x-2">
                          <Button variant="ghost" size="icon" data-testid="button-view-document">
                            <Eye className="h-4 w-4" />
                          </Button>
                          <Button variant="ghost" size="icon" data-testid="button-download-document">
                            <Download className="h-4 w-4" />
                          </Button>
                          <Button variant="ghost" size="icon" data-testid="button-delete-document">
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          ) : (
            <div className="text-center py-12">
              <FileX className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
              <p className="text-lg font-medium mb-2">No documents found</p>
              <p className="text-muted-foreground mb-4">Upload documents to get started</p>
              <Button data-testid="button-upload-first">
                <Upload className="mr-2 h-4 w-4" />
                Upload First Document
              </Button>
            </div>
          )}

          {/* Pagination */}
          {data && data.total > 0 && (
            <div className="flex items-center justify-between pt-4">
              <p className="text-sm text-muted-foreground" data-testid="text-pagination-info">
                Showing {(filters.offset || 0) + 1}-{Math.min((filters.offset || 0) + (filters.limit || 50), data?.total || 0)} of {data?.total || 0} documents
              </p>
              <div className="flex items-center space-x-2">
                <Button 
                  variant="outline" 
                  size="sm" 
                  disabled={!filters.offset || filters.offset === 0}
                  onClick={() => setFilters({...filters, offset: Math.max(0, (filters.offset || 0) - (filters.limit || 50))})}
                  data-testid="button-previous-page"
                >
                  Previous
                </Button>
                <Button 
                  variant="outline" 
                  size="sm"
                  disabled={!data || (filters.offset + filters.limit) >= data.total}
                  onClick={() => setFilters({...filters, offset: (filters.offset || 0) + (filters.limit || 50)})}
                  data-testid="button-next-page"
                >
                  Next
                </Button>
              </div>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
