import { useParams } from "wouter";
import { useEngagement, useEngagementDocuments, useEngagementComments, useEngagementChecklist, useTransitionStatus } from "@/hooks/use-engagements";
import { useClient } from "@/hooks/use-clients";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Skeleton } from "@/components/ui/skeleton";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import WorkflowProgress from "@/components/engagement/workflow-progress";
import Checklist from "@/components/engagement/checklist";
import DocumentsPanel from "@/components/engagement/documents-panel";
import CommentsPanel from "@/components/engagement/comments-panel";
import StatusChip from "@/components/common/status-chip";
import PriorityBadge from "@/components/common/priority-badge";
import { ArrowLeft, Download, Bell, History, FileText } from "lucide-react";
import { Link } from "wouter";

export default function EngagementDetail() {
  const { id } = useParams();
  const { data: engagement, isLoading: engagementLoading } = useEngagement(id!);
  const { data: client, isLoading: clientLoading } = useClient(engagement?.clientId || "");
  const { data: documents } = useEngagementDocuments(id!);
  const { data: comments } = useEngagementComments(id!);
  const { data: checklist } = useEngagementChecklist(id!);
  const transitionStatus = useTransitionStatus();

  const maskPAN = (pan: string) => {
    if (!pan || pan.length !== 10) return pan;
    return pan.substring(0, 2) + "***" + pan.substring(5);
  };

  const formatDueDate = (dueDate: string | Date) => {
    const date = new Date(dueDate);
    const now = new Date();
    const diffInDays = Math.ceil((date.getTime() - now.getTime()) / (1000 * 60 * 60 * 24));
    
    if (diffInDays < 0) {
      return {
        text: date.toLocaleDateString(),
        subtext: `${Math.abs(diffInDays)} days overdue`,
        color: "text-red-600"
      };
    } else if (diffInDays <= 5) {
      return {
        text: date.toLocaleDateString(),
        subtext: `${diffInDays} days remaining`,
        color: "text-orange-600"
      };
    } else {
      return {
        text: date.toLocaleDateString(),
        subtext: `${diffInDays} days remaining`,
        color: "text-green-600"
      };
    }
  };

  const handleStatusTransition = async (newStatus: string, reason?: string) => {
    if (!id) return;
    
    try {
      await transitionStatus.mutateAsync({
        id,
        transition: { status: newStatus, reason }
      });
    } catch (error) {
      console.error("Failed to transition status:", error);
    }
  };

  if (engagementLoading || clientLoading) {
    return (
      <div className="p-6 space-y-6">
        <Skeleton className="h-8 w-48" />
        <Card>
          <CardContent className="p-6">
            <div className="space-y-4">
              <Skeleton className="h-6 w-64" />
              <Skeleton className="h-4 w-full" />
              <Skeleton className="h-32 w-full" />
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  if (!engagement || !client) {
    return (
      <div className="p-6">
        <Card>
          <CardContent className="p-6 text-center">
            <p className="text-muted-foreground">Engagement not found</p>
          </CardContent>
        </Card>
      </div>
    );
  }

  const dueDate = engagement.dueDates && (engagement.dueDates as any).filingDue 
    ? formatDueDate((engagement.dueDates as any).filingDue)
    : null;

  const completedItems = checklist?.filter(item => item.status === "DONE").length || 0;
  const totalItems = checklist?.length || 0;
  const progressPercentage = totalItems > 0 ? (completedItems / totalItems) * 100 : 0;

  return (
    <div className="p-6 space-y-6" data-testid="engagement-detail">
      {/* Header */}
      <div className="flex items-center space-x-4">
        <Link href="/engagements">
          <Button variant="ghost" size="icon" data-testid="button-back">
            <ArrowLeft className="h-4 w-4" />
          </Button>
        </Link>
        <div className="flex-1">
          <div className="flex items-center space-x-3 mb-2">
            <h1 className="text-2xl font-bold" data-testid="text-client-name">{client.displayName}</h1>
            <StatusChip status={engagement.status} />
            <PriorityBadge priority={engagement.priority} />
            {engagement.isAuditApplicable && (
              <Badge variant="outline" data-testid="badge-audit-applicable">Audit Applicable</Badge>
            )}
          </div>
          <div className="flex items-center space-x-6 text-sm text-muted-foreground">
            <span>AY: <span className="font-medium" data-testid="text-ay">{engagement.ay}</span></span>
            <span>Form: <span className="font-medium" data-testid="text-form-type">{engagement.itrFormType}</span></span>
            <span>PAN: <span className="font-medium" data-testid="text-pan">{maskPAN(client.pan)}</span></span>
            {dueDate && (
              <span>Due: <span className={`font-medium ${dueDate.color}`} data-testid="text-due-date">
                {dueDate.text}
              </span></span>
            )}
          </div>
        </div>
        
        <div className="flex items-center space-x-3">
          <Button onClick={() => handleStatusTransition("MANAGER_REVIEW")} data-testid="button-submit-review">
            Submit for Review
          </Button>
          <Button variant="outline" data-testid="button-export-evidence">
            <Download className="mr-2 h-4 w-4" />
            Evidence Pack
          </Button>
        </div>
      </div>

      {/* Main Content */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Main Panel */}
        <div className="lg:col-span-2 space-y-6">
          <Tabs defaultValue="workflow" className="w-full">
            <TabsList className="grid w-full grid-cols-4">
              <TabsTrigger value="workflow" data-testid="tab-workflow">Workflow</TabsTrigger>
              <TabsTrigger value="checklist" data-testid="tab-checklist">Checklist</TabsTrigger>
              <TabsTrigger value="documents" data-testid="tab-documents">Documents</TabsTrigger>
              <TabsTrigger value="comments" data-testid="tab-comments">Comments</TabsTrigger>
            </TabsList>
            
            <TabsContent value="workflow" className="mt-6">
              <WorkflowProgress 
                engagement={engagement} 
                onStatusChange={handleStatusTransition}
              />
            </TabsContent>
            
            <TabsContent value="checklist" className="mt-6">
              <Checklist 
                engagementId={engagement.id}
                items={checklist || []}
                progressPercentage={progressPercentage}
                completedItems={completedItems}
                totalItems={totalItems}
              />
            </TabsContent>
            
            <TabsContent value="documents" className="mt-6">
              <DocumentsPanel 
                engagementId={engagement.id}
                documents={documents || []}
              />
            </TabsContent>
            
            <TabsContent value="comments" className="mt-6">
              <CommentsPanel 
                engagementId={engagement.id}
                comments={comments || []}
              />
            </TabsContent>
          </Tabs>
        </div>

        {/* Sidebar */}
        <div className="space-y-6">
          {/* Assignment Info */}
          <Card>
            <CardHeader>
              <CardTitle className="text-base">Assignment</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="flex justify-between">
                <span className="text-sm text-muted-foreground">Preparer:</span>
                <span className="text-sm font-medium" data-testid="text-preparer">Ankit Patel</span>
              </div>
              <div className="flex justify-between">
                <span className="text-sm text-muted-foreground">Reviewer:</span>
                <span className="text-sm font-medium" data-testid="text-reviewer">Priya Sharma</span>
              </div>
              <div className="flex justify-between">
                <span className="text-sm text-muted-foreground">Partner:</span>
                <span className="text-sm font-medium" data-testid="text-partner">Amit Mehta</span>
              </div>
            </CardContent>
          </Card>

          {/* SLA Status */}
          <Card>
            <CardHeader>
              <CardTitle className="text-base">SLA Status</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="flex justify-between">
                <span className="text-sm text-muted-foreground">Current Stage:</span>
                <span className="text-sm font-medium" data-testid="text-sla-remaining">18h remaining</span>
              </div>
              <div className="w-full h-2 bg-muted rounded-full">
                <div className="h-2 bg-green-500 rounded-full" style={{width: "65%"}}></div>
              </div>
              <div className="flex justify-between text-xs text-muted-foreground">
                <span>Started 3 days ago</span>
                <span>Due in 18h</span>
              </div>
            </CardContent>
          </Card>

          {/* Quick Actions */}
          <Card>
            <CardHeader>
              <CardTitle className="text-base">Quick Actions</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2">
              <Button variant="ghost" className="w-full justify-start" data-testid="button-export-evidence">
                <Download className="mr-2 h-4 w-4" />
                Export Evidence Pack
              </Button>
              <Button variant="ghost" className="w-full justify-start" data-testid="button-send-reminder">
                <Bell className="mr-2 h-4 w-4" />
                Send Reminder
              </Button>
              <Button variant="ghost" className="w-full justify-start" data-testid="button-view-activity">
                <History className="mr-2 h-4 w-4" />
                View Activity Log
              </Button>
              <Button variant="ghost" className="w-full justify-start" data-testid="button-generate-report">
                <FileText className="mr-2 h-4 w-4" />
                Generate Report
              </Button>
            </CardContent>
          </Card>

          {/* Client Info */}
          <Card>
            <CardHeader>
              <CardTitle className="text-base">Client Information</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="flex justify-between">
                <span className="text-sm text-muted-foreground">Legal Name:</span>
                <span className="text-sm font-medium" data-testid="text-legal-name">{client.legalName}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-sm text-muted-foreground">Tier:</span>
                <span className="text-sm font-medium" data-testid="text-client-tier">{engagement.tier}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-sm text-muted-foreground">Risk Category:</span>
                <span className="text-sm font-medium" data-testid="text-risk-category">{client.riskCategory}</span>
              </div>
              {client.aadhaarLast4 && (
                <div className="flex justify-between">
                  <span className="text-sm text-muted-foreground">Aadhaar:</span>
                  <span className="text-sm font-medium font-mono" data-testid="text-aadhaar">****{client.aadhaarLast4}</span>
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
