import { useState } from "react";
import { useEngagements, useCreateEngagement } from "@/hooks/use-engagements";
import { useClients } from "@/hooks/use-clients";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Skeleton } from "@/components/ui/skeleton";
import EngagementForm from "@/components/forms/engagement-form";
import StatusChip from "@/components/common/status-chip";
import PriorityBadge from "@/components/common/priority-badge";
import { Search, Plus, Eye, Edit, Download, Filter } from "lucide-react";
import { Link } from "wouter";
import type { InsertEngagement } from "@shared/schema";

export default function Engagements() {
  const [search, setSearch] = useState("");
  const [isCreateDialogOpen, setIsCreateDialogOpen] = useState(false);
  const [filters, setFilters] = useState({
    q: "",
    status: "",
    priority: "",
    ay: "",
    limit: 50,
    offset: 0
  });

  const { data, isLoading } = useEngagements(filters);
  const createEngagement = useCreateEngagement();

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    setFilters({ ...filters, q: search, offset: 0 });
  };

  const handleCreateEngagement = async (engagementData: InsertEngagement) => {
    try {
      await createEngagement.mutateAsync(engagementData);
      setIsCreateDialogOpen(false);
    } catch (error) {
      console.error("Failed to create engagement:", error);
    }
  };

  const maskPAN = (pan: string) => {
    if (!pan || pan.length !== 10) return pan;
    return pan.substring(0, 2) + "***" + pan.substring(5);
  };

  const formatDueDate = (dueDate: string | Date) => {
    const date = new Date(dueDate);
    const now = new Date();
    const diffInDays = Math.ceil((date.getTime() - now.getTime()) / (1000 * 60 * 60 * 24));
    
    if (diffInDays < 0) {
      return {
        text: date.toLocaleDateString(),
        subtext: `${Math.abs(diffInDays)} days overdue`,
        color: "text-red-600"
      };
    } else if (diffInDays <= 5) {
      return {
        text: date.toLocaleDateString(),
        subtext: `${diffInDays} days remaining`,
        color: "text-orange-600"
      };
    } else {
      return {
        text: date.toLocaleDateString(),
        subtext: `${diffInDays} days remaining`,
        color: "text-green-600"
      };
    }
  };

  if (isLoading) {
    return (
      <div className="p-6 space-y-6">
        <Skeleton className="h-8 w-48" />
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-32" />
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {Array.from({ length: 10 }).map((_, i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6" data-testid="engagements-page">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">Engagements</h1>
          <p className="text-muted-foreground">Manage tax return engagements</p>
        </div>
        
        <Dialog open={isCreateDialogOpen} onOpenChange={setIsCreateDialogOpen}>
          <DialogTrigger asChild>
            <Button data-testid="button-new-engagement">
              <Plus className="mr-2 h-4 w-4" />
              New Engagement
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>Create New Engagement</DialogTitle>
            </DialogHeader>
            <EngagementForm 
              onSubmit={handleCreateEngagement}
              isLoading={createEngagement.isPending}
            />
          </DialogContent>
        </Dialog>
      </div>

      {/* Search and Filters */}
      <Card>
        <CardContent className="p-6">
          <form onSubmit={handleSearch} className="space-y-4">
            <div className="flex items-center space-x-4">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
                <Input
                  type="search"
                  placeholder="Search by client name, PAN, AY..."
                  value={search}
                  onChange={(e) => setSearch(e.target.value)}
                  className="pl-10"
                  data-testid="input-search"
                />
              </div>
              <Button type="submit" data-testid="button-search">Search</Button>
            </div>
            
            <div className="flex items-center space-x-4">
              <div className="flex items-center space-x-2">
                <Filter className="h-4 w-4 text-muted-foreground" />
                <Select value={filters.status} onValueChange={(value) => setFilters({...filters, status: value, offset: 0})}>
                  <SelectTrigger className="w-40">
                    <SelectValue placeholder="All Statuses" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="">All Statuses</SelectItem>
                    <SelectItem value="DATA_REQUESTED">Data Requested</SelectItem>
                    <SelectItem value="COMPUTATION_PREPARED">Computation Prepared</SelectItem>
                    <SelectItem value="MANAGER_REVIEW">Manager Review</SelectItem>
                    <SelectItem value="PARTNER_REVIEW">Partner Review</SelectItem>
                    <SelectItem value="ITR_FILED_ACK_CAPTURED">Filed</SelectItem>
                    <SelectItem value="CLOSED">Closed</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              
              <Select value={filters.priority} onValueChange={(value) => setFilters({...filters, priority: value, offset: 0})}>
                <SelectTrigger className="w-32">
                  <SelectValue placeholder="Priority" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="">All</SelectItem>
                  <SelectItem value="CRITICAL">Critical</SelectItem>
                  <SelectItem value="HIGH">High</SelectItem>
                  <SelectItem value="MEDIUM">Medium</SelectItem>
                  <SelectItem value="LOW">Low</SelectItem>
                </SelectContent>
              </Select>
              
              <Select value={filters.ay} onValueChange={(value) => setFilters({...filters, ay: value, offset: 0})}>
                <SelectTrigger className="w-32">
                  <SelectValue placeholder="AY" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="">All AY</SelectItem>
                  <SelectItem value="2024-25">2024-25</SelectItem>
                  <SelectItem value="2023-24">2023-24</SelectItem>
                  <SelectItem value="2022-23">2022-23</SelectItem>
                </SelectContent>
              </Select>
              
              <Button variant="outline" data-testid="button-export">
                <Download className="mr-2 h-4 w-4" />
                Export CSV
              </Button>
            </div>
          </form>
        </CardContent>
      </Card>

      {/* Engagements Table */}
      <Card>
        <CardHeader>
          <CardTitle>Engagements ({data?.total || 0})</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Client</TableHead>
                  <TableHead>AY</TableHead>
                  <TableHead>Form</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Tier</TableHead>
                  <TableHead>Priority</TableHead>
                  <TableHead>Due Date</TableHead>
                  <TableHead>Created</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {data?.data.map((engagement) => {
                  const dueDate = engagement.dueDates && (engagement.dueDates as any).filingDue 
                    ? formatDueDate((engagement.dueDates as any).filingDue)
                    : null;

                  return (
                    <TableRow key={engagement.id} className="hover:bg-muted/50" data-testid={`engagement-row-${engagement.id}`}>
                      <TableCell>
                        <div>
                          <p className="text-sm font-medium" data-testid="text-client-name">
                            {engagement.client.displayName}
                          </p>
                          <p className="text-xs text-muted-foreground" data-testid="text-client-pan">
                            PAN: {maskPAN(engagement.client.pan)}
                          </p>
                        </div>
                      </TableCell>
                      <TableCell className="text-sm" data-testid="text-ay">{engagement.ay}</TableCell>
                      <TableCell className="text-sm font-medium" data-testid="text-form-type">{engagement.itrFormType}</TableCell>
                      <TableCell>
                        <StatusChip status={engagement.status} />
                      </TableCell>
                      <TableCell className="text-sm" data-testid="text-tier">{engagement.tier}</TableCell>
                      <TableCell>
                        <PriorityBadge priority={engagement.priority} />
                      </TableCell>
                      <TableCell>
                        {dueDate ? (
                          <div>
                            <span className={`text-sm font-medium ${dueDate.color}`} data-testid="text-due-date">
                              {dueDate.text}
                            </span>
                            <p className="text-xs text-muted-foreground" data-testid="text-due-subtext">
                              {dueDate.subtext}
                            </p>
                          </div>
                        ) : (
                          <span className="text-sm text-muted-foreground">No due date</span>
                        )}
                      </TableCell>
                      <TableCell className="text-sm text-muted-foreground" data-testid="text-created-date">
                        {engagement.createdAt ? new Date(engagement.createdAt).toLocaleDateString() : "N/A"}
                      </TableCell>
                      <TableCell className="text-right">
                        <div className="flex justify-end space-x-2">
                          <Link href={`/engagements/${engagement.id}`}>
                            <Button variant="ghost" size="icon" data-testid="button-view-engagement">
                              <Eye className="h-4 w-4" />
                            </Button>
                          </Link>
                          <Button variant="ghost" size="icon" data-testid="button-edit-engagement">
                            <Edit className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  );
                })}
              </TableBody>
            </Table>
          </div>

          {/* Pagination */}
          <div className="flex items-center justify-between pt-4">
            <p className="text-sm text-muted-foreground" data-testid="text-pagination-info">
              Showing {(filters.offset || 0) + 1}-{Math.min((filters.offset || 0) + (filters.limit || 50), data?.total || 0)} of {data?.total || 0} engagements
            </p>
            <div className="flex items-center space-x-2">
              <Button 
                variant="outline" 
                size="sm" 
                disabled={!filters.offset || filters.offset === 0}
                onClick={() => setFilters({...filters, offset: Math.max(0, (filters.offset || 0) - (filters.limit || 50))})}
                data-testid="button-previous-page"
              >
                Previous
              </Button>
              <Button 
                variant="outline" 
                size="sm"
                disabled={!data || (filters.offset + filters.limit) >= data.total}
                onClick={() => setFilters({...filters, offset: (filters.offset || 0) + (filters.limit || 50)})}
                data-testid="button-next-page"
              >
                Next
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
