import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { DatePickerWithRange } from "@/components/ui/date-range-picker";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, PieChart, Pie, Cell, LineChart, Line } from "recharts";
import { Download, TrendingUp, Calendar, Users, FileText, Clock } from "lucide-react";
import type { DashboardStats, PipelineStats } from "@/lib/types";

export default function Reports() {
  const [selectedAY, setSelectedAY] = useState("2024-25");
  const [dateRange, setDateRange] = useState<{from: Date, to: Date} | undefined>(undefined);
  const [reportType, setReportType] = useState("pipeline");

  const { data: stats } = useQuery<DashboardStats>({
    queryKey: ["/api/dashboard/stats", selectedAY],
  });

  const { data: pipeline } = useQuery<PipelineStats[]>({
    queryKey: ["/api/dashboard/pipeline", selectedAY],
  });

  // Mock data for additional reports
  const performanceData = [
    { name: "Ankit Patel", completed: 45, inProgress: 8, avgTAT: 3.2 },
    { name: "Priya Sharma", completed: 38, inProgress: 12, avgTAT: 2.8 },
    { name: "Rohit Singh", completed: 42, inProgress: 6, avgTAT: 3.5 },
    { name: "Kavya Nair", completed: 35, inProgress: 10, avgTAT: 3.1 }
  ];

  const complianceData = [
    { category: "On Time Filing", value: 94.5, target: 95 },
    { category: "E-Verification Rate", value: 98.2, target: 99 },
    { category: "SLA Compliance", value: 87.3, target: 90 },
    { category: "Document Completeness", value: 91.8, target: 95 }
  ];

  const monthlyTrend = [
    { month: "Oct", filed: 45, pending: 23 },
    { month: "Nov", filed: 52, pending: 18 },
    { month: "Dec", filed: 38, pending: 25 },
    { month: "Jan", filed: 67, pending: 15 },
    { month: "Feb", filed: 43, pending: 22 },
    { month: "Mar", filed: 58, pending: 19 }
  ];

  const COLORS = ['#3B82F6', '#10B981', '#F59E0B', '#EF4444', '#8B5CF6'];

  return (
    <div className="p-6 space-y-6" data-testid="reports-page">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">Reports & Analytics</h1>
          <p className="text-muted-foreground">Business intelligence and performance metrics</p>
        </div>
        
        <div className="flex items-center space-x-4">
          <Select value={selectedAY} onValueChange={setSelectedAY}>
            <SelectTrigger className="w-32">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="2024-25">AY 2024-25</SelectItem>
              <SelectItem value="2023-24">AY 2023-24</SelectItem>
              <SelectItem value="2022-23">AY 2022-23</SelectItem>
            </SelectContent>
          </Select>
          
          <Button variant="outline" data-testid="button-export-reports">
            <Download className="mr-2 h-4 w-4" />
            Export Reports
          </Button>
        </div>
      </div>

      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                <FileText className="h-6 w-6 text-blue-600" />
              </div>
              <div>
                <p className="text-2xl font-bold" data-testid="text-total-engagements">{stats?.totalEngagements || 0}</p>
                <p className="text-sm text-muted-foreground">Total Engagements</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                <TrendingUp className="h-6 w-6 text-green-600" />
              </div>
              <div>
                <p className="text-2xl font-bold text-green-600" data-testid="text-filed-returns">{stats?.filedReturns || 0}</p>
                <p className="text-sm text-muted-foreground">Filed Returns</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
                <Clock className="h-6 w-6 text-orange-600" />
              </div>
              <div>
                <p className="text-2xl font-bold text-orange-600" data-testid="text-pending-reviews">{stats?.pendingReviews || 0}</p>
                <p className="text-sm text-muted-foreground">Pending Reviews</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-red-100 rounded-lg flex items-center justify-center">
                <Users className="h-6 w-6 text-red-600" />
              </div>
              <div>
                <p className="text-2xl font-bold text-red-600" data-testid="text-sla-breaches">{stats?.slaBreaches || 0}</p>
                <p className="text-sm text-muted-foreground">SLA Breaches</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Report Tabs */}
      <Tabs value={reportType} onValueChange={setReportType} className="w-full">
        <TabsList className="grid w-full grid-cols-4">
          <TabsTrigger value="pipeline" data-testid="tab-pipeline">Pipeline Analysis</TabsTrigger>
          <TabsTrigger value="performance" data-testid="tab-performance">Team Performance</TabsTrigger>
          <TabsTrigger value="compliance" data-testid="tab-compliance">Compliance</TabsTrigger>
          <TabsTrigger value="trends" data-testid="tab-trends">Trends</TabsTrigger>
        </TabsList>

        <TabsContent value="pipeline" className="mt-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle>Pipeline Distribution</CardTitle>
              </CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={300}>
                  <PieChart>
                    <Pie
                      data={pipeline}
                      cx="50%"
                      cy="50%"
                      outerRadius={80}
                      dataKey="count"
                      nameKey="status"
                    >
                      {pipeline?.map((entry, index) => (
                        <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                      ))}
                    </Pie>
                    <Tooltip />
                  </PieChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Status Breakdown</CardTitle>
              </CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={300}>
                  <BarChart data={pipeline}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="status" />
                    <YAxis />
                    <Tooltip />
                    <Bar dataKey="count" fill="#3B82F6" />
                  </BarChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="performance" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle>Team Performance Metrics</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead>
                    <tr className="border-b">
                      <th className="text-left py-3 px-4">Team Member</th>
                      <th className="text-left py-3 px-4">Completed</th>
                      <th className="text-left py-3 px-4">In Progress</th>
                      <th className="text-left py-3 px-4">Avg TAT (days)</th>
                      <th className="text-left py-3 px-4">Performance</th>
                    </tr>
                  </thead>
                  <tbody>
                    {performanceData.map((member, index) => (
                      <tr key={index} className="border-b hover:bg-muted/50" data-testid={`performance-row-${index}`}>
                        <td className="py-3 px-4 font-medium" data-testid="text-member-name">{member.name}</td>
                        <td className="py-3 px-4" data-testid="text-completed">{member.completed}</td>
                        <td className="py-3 px-4" data-testid="text-in-progress">{member.inProgress}</td>
                        <td className="py-3 px-4" data-testid="text-avg-tat">{member.avgTAT}</td>
                        <td className="py-3 px-4">
                          <Badge variant={member.avgTAT <= 3 ? "default" : "destructive"} data-testid="badge-performance">
                            {member.avgTAT <= 3 ? "Good" : "Needs Improvement"}
                          </Badge>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="compliance" className="mt-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle>Compliance Metrics</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {complianceData.map((metric, index) => (
                  <div key={index} className="space-y-2" data-testid={`compliance-metric-${index}`}>
                    <div className="flex justify-between items-center">
                      <span className="text-sm font-medium" data-testid="text-metric-name">{metric.category}</span>
                      <span className={`text-sm font-bold ${metric.value >= metric.target ? "text-green-600" : "text-red-600"}`} data-testid="text-metric-value">
                        {metric.value}%
                      </span>
                    </div>
                    <div className="w-full h-2 bg-muted rounded-full">
                      <div 
                        className={`h-2 rounded-full ${metric.value >= metric.target ? "bg-green-500" : "bg-red-500"}`}
                        style={{width: `${metric.value}%`}}
                      ></div>
                    </div>
                    <div className="flex justify-between text-xs text-muted-foreground">
                      <span>Target: {metric.target}%</span>
                      <span>{metric.value >= metric.target ? "✓ Met" : "⚠ Below Target"}</span>
                    </div>
                  </div>
                ))}
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Regulatory Compliance</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <span className="text-sm">DPDP Act 2023</span>
                  <Badge className="bg-green-100 text-green-800" data-testid="badge-dpdp-compliance">
                    ✓ Compliant
                  </Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm">IT Act 2000 (Sec 65B)</span>
                  <Badge className="bg-green-100 text-green-800" data-testid="badge-it-act-compliance">
                    ✓ Compliant
                  </Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm">ICAI Code of Ethics</span>
                  <Badge className="bg-green-100 text-green-800" data-testid="badge-icai-compliance">
                    ✓ Compliant
                  </Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm">Data Retention Policy</span>
                  <Badge className="bg-green-100 text-green-800" data-testid="badge-retention-compliance">
                    ✓ Compliant
                  </Badge>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="trends" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle>Monthly Filing Trends</CardTitle>
            </CardHeader>
            <CardContent>
              <ResponsiveContainer width="100%" height={400}>
                <LineChart data={monthlyTrend}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="month" />
                  <YAxis />
                  <Tooltip />
                  <Line type="monotone" dataKey="filed" stroke="#10B981" strokeWidth={2} name="Filed" />
                  <Line type="monotone" dataKey="pending" stroke="#F59E0B" strokeWidth={2} name="Pending" />
                </LineChart>
              </ResponsiveContainer>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      {/* Export Options */}
      <Card>
        <CardHeader>
          <CardTitle>Export Options</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <Button variant="outline" className="h-20 flex flex-col space-y-2" data-testid="button-export-pipeline">
              <FileText className="h-6 w-6" />
              <span>Pipeline Report</span>
            </Button>
            <Button variant="outline" className="h-20 flex flex-col space-y-2" data-testid="button-export-performance">
              <TrendingUp className="h-6 w-6" />
              <span>Performance Report</span>
            </Button>
            <Button variant="outline" className="h-20 flex flex-col space-y-2" data-testid="button-export-compliance">
              <Calendar className="h-6 w-6" />
              <span>Compliance Report</span>
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
