import { useState } from "react";
import { useEngagements } from "@/hooks/use-engagements";
import { useAuth } from "@/lib/auth";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import ReviewCard from "@/components/review/review-card";
import StatusChip from "@/components/common/status-chip";
import { ClipboardCheck, Clock, AlertTriangle } from "lucide-react";

export default function Reviews() {
  const { user } = useAuth();
  const [selectedAY, setSelectedAY] = useState("2024-25");

  // Get engagements pending review based on user role
  const getReviewStatuses = () => {
    switch (user?.role) {
      case "MANAGER":
        return ["MANAGER_REVIEW"];
      case "PARTNER":
        return ["PARTNER_REVIEW"];
      default:
        return ["MANAGER_REVIEW", "PARTNER_REVIEW"];
    }
  };

  const { data: pendingReviews } = useEngagements({
    status: getReviewStatuses().join(","),
    ay: selectedAY,
    limit: 50
  });

  const { data: allReviews } = useEngagements({
    status: "MANAGER_REVIEW,PARTNER_REVIEW,CLIENT_REVIEW_SIGNOFF",
    limit: 100
  });

  const getReviewTypeFromStatus = (status: string) => {
    switch (status) {
      case "MANAGER_REVIEW":
        return "Manager Review";
      case "PARTNER_REVIEW":
        return "Partner Review";
      case "CLIENT_REVIEW_SIGNOFF":
        return "Client Sign-off";
      default:
        return "Review";
    }
  };

  const isOverdue = (engagement: any) => {
    if (!engagement.dueDates?.filingDue) return false;
    const dueDate = new Date(engagement.dueDates.filingDue);
    return dueDate < new Date();
  };

  const getDaysUntilDue = (engagement: any) => {
    if (!engagement.dueDates?.filingDue) return null;
    const dueDate = new Date(engagement.dueDates.filingDue);
    const today = new Date();
    const diffInDays = Math.ceil((dueDate.getTime() - today.getTime()) / (1000 * 60 * 60 * 24));
    
    if (diffInDays < 0) return `${Math.abs(diffInDays)} days overdue`;
    if (diffInDays === 0) return "Due today";
    if (diffInDays === 1) return "Due tomorrow";
    return `${diffInDays} days remaining`;
  };

  const myReviews = pendingReviews?.data.filter(eng => {
    if (user?.role === "MANAGER") return eng.status === "MANAGER_REVIEW";
    if (user?.role === "PARTNER") return eng.status === "PARTNER_REVIEW";
    return true;
  }) || [];

  const urgentReviews = allReviews?.data.filter(isOverdue) || [];

  return (
    <div className="p-6 space-y-6" data-testid="reviews-page">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">Review Queue</h1>
          <p className="text-muted-foreground">Manage pending reviews and approvals</p>
        </div>
        
        <div className="flex items-center space-x-4">
          <div className="flex items-center space-x-2">
            <span className="text-sm text-muted-foreground">Role:</span>
            <Badge variant="default" data-testid="badge-user-role">
              {user?.role || "Unknown"}
            </Badge>
          </div>
          <Select value={selectedAY} onValueChange={setSelectedAY}>
            <SelectTrigger className="w-32">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="2024-25">AY 2024-25</SelectItem>
              <SelectItem value="2023-24">AY 2023-24</SelectItem>
              <SelectItem value="2022-23">AY 2022-23</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                <ClipboardCheck className="h-6 w-6 text-blue-600" />
              </div>
              <div>
                <p className="text-2xl font-bold" data-testid="text-my-reviews-count">{myReviews.length}</p>
                <p className="text-sm text-muted-foreground">My Reviews</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
                <Clock className="h-6 w-6 text-orange-600" />
              </div>
              <div>
                <p className="text-2xl font-bold" data-testid="text-urgent-count">{urgentReviews.length}</p>
                <p className="text-sm text-muted-foreground">Urgent</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-red-100 rounded-lg flex items-center justify-center">
                <AlertTriangle className="h-6 w-6 text-red-600" />
              </div>
              <div>
                <p className="text-2xl font-bold text-red-600" data-testid="text-overdue-count">
                  {urgentReviews.length}
                </p>
                <p className="text-sm text-muted-foreground">Overdue</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Review Tabs */}
      <Tabs defaultValue="my-reviews" className="w-full">
        <TabsList>
          <TabsTrigger value="my-reviews" data-testid="tab-my-reviews">My Reviews</TabsTrigger>
          <TabsTrigger value="all-reviews" data-testid="tab-all-reviews">All Reviews</TabsTrigger>
          <TabsTrigger value="urgent" data-testid="tab-urgent">Urgent</TabsTrigger>
        </TabsList>

        <TabsContent value="my-reviews" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle>My Reviews ({myReviews.length})</CardTitle>
            </CardHeader>
            <CardContent>
              {myReviews.length > 0 ? (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                  {myReviews.map((engagement) => (
                    <ReviewCard 
                      key={engagement.id}
                      item={{
                        id: engagement.id,
                        clientName: engagement.client.displayName,
                        details: `${engagement.itrFormType} • ${engagement.ay}`,
                        stage: getReviewTypeFromStatus(engagement.status),
                        preparer: "Staff Member", // Would come from assignments
                        reviewer: user?.role === "MANAGER" ? "You" : "Manager",
                        due: getDaysUntilDue(engagement) || "No due date"
                      }}
                    />
                  ))}
                </div>
              ) : (
                <div className="text-center py-8">
                  <ClipboardCheck className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">No reviews pending</p>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="all-reviews" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle>All Pending Reviews ({allReviews?.total || 0})</CardTitle>
            </CardHeader>
            <CardContent>
              {allReviews?.data && allReviews.data.length > 0 ? (
                <div className="space-y-4">
                  {allReviews.data.map((engagement) => (
                    <div key={engagement.id} className="flex items-center justify-between p-4 border border-border rounded-lg hover:bg-muted/50" data-testid={`review-item-${engagement.id}`}>
                      <div className="flex items-center space-x-4">
                        <div>
                          <p className="font-medium" data-testid="text-engagement-client">{engagement.client.displayName}</p>
                          <p className="text-sm text-muted-foreground">{engagement.itrFormType} • {engagement.ay}</p>
                        </div>
                        <StatusChip status={engagement.status} />
                        <PriorityBadge priority={engagement.priority} />
                      </div>
                      
                      <div className="flex items-center space-x-4">
                        <div className="text-right">
                          <p className="text-sm font-medium" data-testid="text-days-until-due">
                            {getDaysUntilDue(engagement)}
                          </p>
                          <p className="text-xs text-muted-foreground">
                            {getReviewTypeFromStatus(engagement.status)}
                          </p>
                        </div>
                        <Button size="sm" data-testid="button-review">
                          Review
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-8">
                  <ClipboardCheck className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">No reviews pending</p>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="urgent" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <AlertTriangle className="h-5 w-5 text-red-600" />
                <span>Urgent Reviews ({urgentReviews.length})</span>
              </CardTitle>
            </CardHeader>
            <CardContent>
              {urgentReviews.length > 0 ? (
                <div className="space-y-4">
                  {urgentReviews.map((engagement) => (
                    <div key={engagement.id} className="flex items-center justify-between p-4 border border-red-200 bg-red-50 rounded-lg" data-testid={`urgent-review-${engagement.id}`}>
                      <div className="flex items-center space-x-4">
                        <AlertTriangle className="h-5 w-5 text-red-600" />
                        <div>
                          <p className="font-medium text-red-900" data-testid="text-urgent-client">{engagement.client.displayName}</p>
                          <p className="text-sm text-red-700">{engagement.itrFormType} • {engagement.ay}</p>
                        </div>
                        <StatusChip status={engagement.status} />
                      </div>
                      
                      <div className="flex items-center space-x-4">
                        <div className="text-right">
                          <p className="text-sm font-medium text-red-600" data-testid="text-overdue-days">
                            {getDaysUntilDue(engagement)}
                          </p>
                          <p className="text-xs text-red-500">
                            {getReviewTypeFromStatus(engagement.status)}
                          </p>
                        </div>
                        <Button size="sm" variant="destructive" data-testid="button-urgent-review">
                          Review Now
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-8">
                  <AlertTriangle className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">No urgent reviews</p>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
