import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { useAuth } from "@/lib/auth";
import { Settings as SettingsIcon, Users, Bell, Shield, Database, Mail } from "lucide-react";
import type { User } from "@shared/schema";

export default function Settings() {
  const { user } = useAuth();
  const [activeTab, setActiveTab] = useState("users");

  const { data: users } = useQuery<User[]>({
    queryKey: ["/api/users"],
  });

  const slaProfiles = [
    { id: "1", name: "Standard Individual", stages: { "COMPUTATION_PREPARED": 24, "MANAGER_REVIEW": 12, "PARTNER_REVIEW": 8 } },
    { id: "2", name: "Corporate", stages: { "COMPUTATION_PREPARED": 48, "MANAGER_REVIEW": 24, "PARTNER_REVIEW": 12 } },
    { id: "3", name: "Audit Cases", stages: { "COMPUTATION_PREPARED": 72, "MANAGER_REVIEW": 36, "PARTNER_REVIEW": 24 } }
  ];

  const reminderTemplates = [
    { key: "DATA_REQUEST", name: "Data Request (Client)", channel: "EMAIL" },
    { key: "REVIEW_DUE", name: "Review Due (Manager)", channel: "EMAIL" },
    { key: "E_VERIFY_REMINDER", name: "E-Verification Reminder", channel: "EMAIL" },
    { key: "SLA_BREACH", name: "SLA Breach Alert", channel: "EMAIL" }
  ];

  const getUserInitials = (name: string) => {
    return name
      .split(" ")
      .map(n => n[0])
      .join("")
      .toUpperCase()
      .substring(0, 2);
  };

  const getRoleBadgeColor = (role: string) => {
    switch (role) {
      case "PARTNER":
        return "bg-purple-100 text-purple-800";
      case "MANAGER":
        return "bg-orange-100 text-orange-800";
      case "STAFF":
        return "bg-blue-100 text-blue-800";
      case "CLIENT":
        return "bg-green-100 text-green-800";
      case "AUDITOR":
        return "bg-gray-100 text-gray-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  if (user?.role !== "PARTNER" && user?.role !== "MANAGER") {
    return (
      <div className="p-6">
        <Card>
          <CardContent className="p-6 text-center">
            <Shield className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
            <p className="text-lg font-medium mb-2">Access Denied</p>
            <p className="text-muted-foreground">You don't have permission to access settings</p>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6" data-testid="settings-page">
      {/* Header */}
      <div>
        <h1 className="text-2xl font-bold flex items-center space-x-3" data-testid="text-page-title">
          <SettingsIcon className="h-6 w-6" />
          <span>Settings</span>
        </h1>
        <p className="text-muted-foreground">Configure system settings and preferences</p>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
        <TabsList className="grid w-full grid-cols-5">
          <TabsTrigger value="users" data-testid="tab-users">Users</TabsTrigger>
          <TabsTrigger value="sla" data-testid="tab-sla">SLA Profiles</TabsTrigger>
          <TabsTrigger value="reminders" data-testid="tab-reminders">Reminders</TabsTrigger>
          <TabsTrigger value="notifications" data-testid="tab-notifications">Notifications</TabsTrigger>
          <TabsTrigger value="system" data-testid="tab-system">System</TabsTrigger>
        </TabsList>

        <TabsContent value="users" className="mt-6">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="flex items-center space-x-2">
                  <Users className="h-5 w-5" />
                  <span>User Management</span>
                </CardTitle>
                <Button data-testid="button-invite-user">Invite User</Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {users?.map((user) => (
                  <div key={user.id} className="flex items-center justify-between p-4 border border-border rounded-lg" data-testid={`user-item-${user.id}`}>
                    <div className="flex items-center space-x-4">
                      <div className="w-10 h-10 bg-secondary rounded-full flex items-center justify-center">
                        <span className="text-secondary-foreground font-medium" data-testid="text-user-initials">
                          {getUserInitials(user.name)}
                        </span>
                      </div>
                      <div>
                        <p className="font-medium" data-testid="text-user-name">{user.name}</p>
                        <p className="text-sm text-muted-foreground" data-testid="text-user-email">{user.email}</p>
                      </div>
                      <Badge className={getRoleBadgeColor(user.role)} data-testid="badge-user-role">
                        {user.role}
                      </Badge>
                      <Badge variant={user.isActive ? "default" : "secondary"} data-testid="badge-user-status">
                        {user.isActive ? "Active" : "Inactive"}
                      </Badge>
                    </div>
                    
                    <div className="flex items-center space-x-2">
                      <Button variant="ghost" size="icon" data-testid="button-edit-user">
                        <SettingsIcon className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="sla" className="mt-6">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle>SLA Profiles</CardTitle>
                <Button data-testid="button-new-sla-profile">New Profile</Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {slaProfiles.map((profile) => (
                  <div key={profile.id} className="p-4 border border-border rounded-lg" data-testid={`sla-profile-${profile.id}`}>
                    <div className="flex items-center justify-between mb-3">
                      <h4 className="font-medium" data-testid="text-profile-name">{profile.name}</h4>
                      <Button variant="ghost" size="sm" data-testid="button-edit-profile">Edit</Button>
                    </div>
                    <div className="grid grid-cols-3 gap-4 text-sm">
                      <div>
                        <span className="text-muted-foreground">Computation:</span>
                        <span className="ml-2 font-medium" data-testid="text-computation-hours">{profile.stages.COMPUTATION_PREPARED}h</span>
                      </div>
                      <div>
                        <span className="text-muted-foreground">Manager Review:</span>
                        <span className="ml-2 font-medium" data-testid="text-manager-hours">{profile.stages.MANAGER_REVIEW}h</span>
                      </div>
                      <div>
                        <span className="text-muted-foreground">Partner Review:</span>
                        <span className="ml-2 font-medium" data-testid="text-partner-hours">{profile.stages.PARTNER_REVIEW}h</span>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="reminders" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle>Reminder Templates</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {reminderTemplates.map((template, index) => (
                  <div key={template.key} className="p-4 border border-border rounded-lg" data-testid={`reminder-template-${index}`}>
                    <div className="flex items-center justify-between mb-2">
                      <h4 className="font-medium" data-testid="text-template-name">{template.name}</h4>
                      <div className="flex items-center space-x-2">
                        <Badge variant="outline" data-testid="badge-template-channel">{template.channel}</Badge>
                        <Button variant="ghost" size="sm" data-testid="button-edit-template">Edit</Button>
                      </div>
                    </div>
                    <p className="text-sm text-muted-foreground" data-testid="text-template-key">Key: {template.key}</p>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="notifications" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <Bell className="h-5 w-5" />
                <span>Notification Settings</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base">Email Notifications</Label>
                    <p className="text-sm text-muted-foreground">Receive notifications via email</p>
                  </div>
                  <Switch defaultChecked data-testid="switch-email-notifications" />
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base">WhatsApp Notifications</Label>
                    <p className="text-sm text-muted-foreground">Receive notifications via WhatsApp</p>
                  </div>
                  <Switch data-testid="switch-whatsapp-notifications" />
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base">SLA Breach Alerts</Label>
                    <p className="text-sm text-muted-foreground">Get alerted when SLAs are breached</p>
                  </div>
                  <Switch defaultChecked data-testid="switch-sla-alerts" />
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base">Daily Summary</Label>
                    <p className="text-sm text-muted-foreground">Receive daily engagement summary</p>
                  </div>
                  <Switch defaultChecked data-testid="switch-daily-summary" />
                </div>
              </div>

              <Separator />

              <div className="space-y-4">
                <h4 className="font-medium">Email Configuration</h4>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="smtp-host">SMTP Host</Label>
                    <Input 
                      id="smtp-host" 
                      placeholder="smtp.gmail.com"
                      defaultValue={import.meta.env.VITE_SMTP_HOST || ""}
                      data-testid="input-smtp-host"
                    />
                  </div>
                  <div>
                    <Label htmlFor="smtp-port">SMTP Port</Label>
                    <Input 
                      id="smtp-port" 
                      placeholder="587"
                      defaultValue={import.meta.env.VITE_SMTP_PORT || ""}
                      data-testid="input-smtp-port"
                    />
                  </div>
                  <div>
                    <Label htmlFor="smtp-user">SMTP Username</Label>
                    <Input 
                      id="smtp-user" 
                      placeholder="your-email@domain.com"
                      data-testid="input-smtp-user"
                    />
                  </div>
                  <div>
                    <Label htmlFor="smtp-pass">SMTP Password</Label>
                    <Input 
                      id="smtp-pass" 
                      type="password" 
                      placeholder="••••••••"
                      data-testid="input-smtp-password"
                    />
                  </div>
                </div>
              </div>

              <div className="space-y-4">
                <h4 className="font-medium">WhatsApp Configuration</h4>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="whatsapp-api-key">API Key</Label>
                    <Input 
                      id="whatsapp-api-key" 
                      placeholder="Your WhatsApp API key"
                      data-testid="input-whatsapp-key"
                    />
                  </div>
                  <div>
                    <Label htmlFor="whatsapp-endpoint">Endpoint URL</Label>
                    <Input 
                      id="whatsapp-endpoint" 
                      placeholder="https://api.wati.io/api/v1/"
                      data-testid="input-whatsapp-endpoint"
                    />
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="system" className="mt-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Database className="h-5 w-5" />
                  <span>System Configuration</span>
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="jwt-expiry">JWT Token Expiry (days)</Label>
                  <Input 
                    id="jwt-expiry" 
                    type="number" 
                    defaultValue="7"
                    data-testid="input-jwt-expiry"
                  />
                </div>
                
                <div>
                  <Label htmlFor="max-file-size">Max File Size (MB)</Label>
                  <Input 
                    id="max-file-size" 
                    type="number" 
                    defaultValue="10"
                    data-testid="input-max-file-size"
                  />
                </div>
                
                <div>
                  <Label htmlFor="session-timeout">Session Timeout (minutes)</Label>
                  <Input 
                    id="session-timeout" 
                    type="number" 
                    defaultValue="60"
                    data-testid="input-session-timeout"
                  />
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base">Enable 2FA</Label>
                    <p className="text-sm text-muted-foreground">Require two-factor authentication</p>
                  </div>
                  <Switch data-testid="switch-2fa" />
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-base">Audit All Actions</Label>
                    <p className="text-sm text-muted-foreground">Log all user actions for compliance</p>
                  </div>
                  <Switch defaultChecked disabled data-testid="switch-audit-actions" />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Data Retention Policy</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-3">
                  <div className="flex justify-between items-center">
                    <span className="text-sm">Client Records</span>
                    <span className="font-medium" data-testid="text-client-retention">8 years</span>
                  </div>
                  <div className="flex justify-between items-center">
                    <span className="text-sm">Activity Logs</span>
                    <span className="font-medium" data-testid="text-logs-retention">10 years</span>
                  </div>
                  <div className="flex justify-between items-center">
                    <span className="text-sm">Documents</span>
                    <span className="font-medium" data-testid="text-documents-retention">8 years</span>
                  </div>
                  <div className="flex justify-between items-center">
                    <span className="text-sm">Computation Records</span>
                    <span className="font-medium" data-testid="text-computation-retention">8 years</span>
                  </div>
                </div>
                
                <Separator />
                
                <div className="space-y-3">
                  <h5 className="font-medium">Automatic Cleanup</h5>
                  <div className="flex items-center justify-between">
                    <span className="text-sm">Enable auto-cleanup</span>
                    <Switch data-testid="switch-auto-cleanup" />
                  </div>
                  <div>
                    <Label htmlFor="cleanup-frequency">Cleanup Frequency (days)</Label>
                    <Input 
                      id="cleanup-frequency" 
                      type="number" 
                      defaultValue="30"
                      data-testid="input-cleanup-frequency"
                    />
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Backup & Export</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <Button variant="outline" className="h-20 flex flex-col space-y-2" data-testid="button-backup-database">
                  <Database className="h-6 w-6" />
                  <span>Backup Database</span>
                </Button>
                <Button variant="outline" className="h-20 flex flex-col space-y-2" data-testid="button-export-audit">
                  <Shield className="h-6 w-6" />
                  <span>Export Audit Log</span>
                </Button>
                <Button variant="outline" className="h-20 flex flex-col space-y-2" data-testid="button-system-health">
                  <SettingsIcon className="h-6 w-6" />
                  <span>System Health</span>
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
