-- ITR Tracker Initial Schema Migration
-- Creates all tables required for the application

-- Enable UUID extension
CREATE EXTENSION IF NOT EXISTS "uuid-ossp";

-- Create enums
CREATE TYPE user_role AS ENUM ('PARTNER', 'MANAGER', 'STAFF', 'CLIENT', 'AUDITOR');
CREATE TYPE engagement_status AS ENUM (
  'DATA_REQUESTED',
  'DATA_RECEIVED_PRELIMINARY', 
  'KYC_CONSENT_CAPTURED',
  'DOC_INDEXING_VETTING',
  'AIS_TIS_IMPORTED_RECONCILED',
  'COMPUTATION_PREPARED',
  'MANAGER_REVIEW',
  'PARTNER_REVIEW', 
  'CLIENT_REVIEW_SIGNOFF',
  'ITR_CREATED_JSON_VALIDATED',
  'ITR_FILED_ACK_CAPTURED',
  'E_VERIFICATION_PENDING',
  'E_VERIFICATION_DONE',
  'CPC_PROCESSING',
  'REFUND_ADJUSTMENT_UPDATE',
  'CLOSED'
);
CREATE TYPE priority AS ENUM ('LOW', 'MEDIUM', 'HIGH', 'CRITICAL');
CREATE TYPE itr_form_type AS ENUM ('ITR-1', 'ITR-2', 'ITR-3', 'ITR-4', 'ITR-5', 'ITR-6', 'ITR-7');
CREATE TYPE tier AS ENUM ('INDIVIDUAL', 'HUF', 'FIRM', 'COMPANY', 'TRUST');
CREATE TYPE checklist_status AS ENUM ('PENDING', 'IN_PROGRESS', 'DONE', 'NA');
CREATE TYPE review_status AS ENUM ('PENDING', 'APPROVED', 'REJECTED', 'CHANGES_REQUESTED');
CREATE TYPE e_verify_status AS ENUM ('PENDING', 'DONE', 'BY_AADHAAR_OTP', 'BY_BANK', 'BY_DSC');
CREATE TYPE cpc_status AS ENUM ('PROCESSED', 'DEFECTIVE', 'NOTICE_ISSUED');
CREATE TYPE refund_status AS ENUM ('ISSUED', 'PENDING', 'ADJUSTED');
CREATE TYPE reminder_channel AS ENUM ('EMAIL', 'WHATSAPP', 'SMS');
CREATE TYPE reminder_status AS ENUM ('SCHEDULED', 'SENT', 'FAILED');

-- Users table
CREATE TABLE users (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    name TEXT NOT NULL,
    email TEXT NOT NULL UNIQUE,
    mobile TEXT,
    password TEXT NOT NULL,
    role user_role NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    last_login_at TIMESTAMP WITH TIME ZONE,
    two_fa_secret TEXT,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Clients table
CREATE TABLE clients (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    display_name TEXT NOT NULL,
    legal_name TEXT NOT NULL,
    pan TEXT NOT NULL UNIQUE,
    aadhaar_last4 TEXT,
    date_of_birth TIMESTAMP WITH TIME ZONE,
    incorporation_date TIMESTAMP WITH TIME ZONE,
    address JSONB,
    contacts JSONB DEFAULT '[]',
    risk_category TEXT DEFAULT 'MEDIUM',
    tags JSONB DEFAULT '[]',
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Engagements table
CREATE TABLE engagements (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    client_id UUID REFERENCES clients(id) NOT NULL,
    ay TEXT NOT NULL,
    itr_form_type itr_form_type NOT NULL,
    status engagement_status DEFAULT 'DATA_REQUESTED',
    tier tier NOT NULL,
    is_audit_applicable BOOLEAN DEFAULT FALSE,
    due_dates JSONB DEFAULT '{}',
    sla_profile_id UUID,
    priority priority DEFAULT 'MEDIUM',
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Assignments table
CREATE TABLE assignments (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    engagement_id UUID REFERENCES engagements(id) NOT NULL,
    user_id UUID REFERENCES users(id) NOT NULL,
    role_on_case TEXT NOT NULL,
    start_date TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
    end_date TIMESTAMP WITH TIME ZONE,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Documents table
CREATE TABLE documents (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    engagement_id UUID REFERENCES engagements(id) NOT NULL,
    doc_type TEXT NOT NULL,
    file_name TEXT NOT NULL,
    file_path TEXT NOT NULL,
    file_size INTEGER NOT NULL,
    checksum TEXT NOT NULL,
    uploaded_by UUID REFERENCES users(id) NOT NULL,
    client_visible BOOLEAN DEFAULT FALSE,
    version INTEGER DEFAULT 1,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Checklists table
CREATE TABLE checklists (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    engagement_id UUID REFERENCES engagements(id) NOT NULL,
    name TEXT NOT NULL,
    version TEXT DEFAULT '1.0',
    is_template BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Checklist items table
CREATE TABLE checklist_items (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    checklist_id UUID REFERENCES checklists(id) NOT NULL,
    item TEXT NOT NULL,
    section_ref TEXT,
    mandatory BOOLEAN DEFAULT FALSE,
    status checklist_status DEFAULT 'PENDING',
    assignee_id UUID REFERENCES users(id),
    due_at TIMESTAMP WITH TIME ZONE,
    completed_at TIMESTAMP WITH TIME ZONE,
    notes TEXT,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Computations table
CREATE TABLE computations (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    engagement_id UUID REFERENCES engagements(id) NOT NULL,
    engine TEXT NOT NULL,
    version TEXT DEFAULT '1.0',
    summary_json JSONB NOT NULL,
    attachments JSONB DEFAULT '[]',
    prepared_by UUID REFERENCES users(id) NOT NULL,
    reviewed_by UUID REFERENCES users(id),
    review_status review_status DEFAULT 'PENDING',
    remarks TEXT,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- E-filing table
CREATE TABLE efiling (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    engagement_id UUID REFERENCES engagements(id) NOT NULL,
    json_uploaded BOOLEAN DEFAULT FALSE,
    ack_no TEXT,
    filed_at TIMESTAMP WITH TIME ZONE,
    e_verify_status e_verify_status DEFAULT 'PENDING',
    e_verified_at TIMESTAMP WITH TIME ZONE,
    cpc_status cpc_status,
    refund_status refund_status,
    refund_amount DECIMAL,
    cpc_comm_ref TEXT,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Comments table
CREATE TABLE comments (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    engagement_id UUID REFERENCES engagements(id) NOT NULL,
    entity_type TEXT NOT NULL,
    entity_id UUID NOT NULL,
    author_id UUID REFERENCES users(id) NOT NULL,
    body TEXT NOT NULL,
    mentions JSONB DEFAULT '[]',
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Reminders table
CREATE TABLE reminders (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    engagement_id UUID REFERENCES engagements(id) NOT NULL,
    channel reminder_channel NOT NULL,
    template_key TEXT NOT NULL,
    schedule_at TIMESTAMP WITH TIME ZONE NOT NULL,
    sent_at TIMESTAMP WITH TIME ZONE,
    status reminder_status DEFAULT 'SCHEDULED',
    payload_json JSONB NOT NULL,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- SLA profiles table
CREATE TABLE sla_profiles (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    name TEXT NOT NULL,
    stage_targets_json JSONB NOT NULL,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- SLA events table
CREATE TABLE sla_events (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    engagement_id UUID REFERENCES engagements(id) NOT NULL,
    stage TEXT NOT NULL,
    breached BOOLEAN DEFAULT FALSE,
    breached_at TIMESTAMP WITH TIME ZONE,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Activity log table (append-only, immutable)
CREATE TABLE activity_log (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    actor_user_id UUID REFERENCES users(id),
    actor_role TEXT NOT NULL,
    action TEXT NOT NULL,
    entity_type TEXT NOT NULL,
    entity_id UUID NOT NULL,
    ip TEXT NOT NULL,
    user_agent TEXT,
    before_json JSONB,
    after_json JSONB,
    signature_hash TEXT NOT NULL,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Status transitions table
CREATE TABLE status_transitions (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    engagement_id UUID REFERENCES engagements(id) NOT NULL,
    from_status TEXT,
    to_status TEXT NOT NULL,
    reason TEXT,
    actor_user_id UUID REFERENCES users(id) NOT NULL,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Tags table
CREATE TABLE tags (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    name TEXT NOT NULL UNIQUE,
    color TEXT DEFAULT '#6B7280',
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Entity tags table
CREATE TABLE entity_tags (
    id UUID PRIMARY KEY DEFAULT uuid_generate_v4(),
    entity_type TEXT NOT NULL,
    entity_id UUID NOT NULL,
    tag_id UUID REFERENCES tags(id) NOT NULL,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
    UNIQUE(entity_type, entity_id, tag_id)
);

-- Create indexes for performance
CREATE INDEX idx_users_email ON users(email);
CREATE INDEX idx_users_role ON users(role);
CREATE INDEX idx_clients_pan ON clients(pan);
CREATE INDEX idx_engagements_client_id ON engagements(client_id);
CREATE INDEX idx_engagements_status ON engagements(status);
CREATE INDEX idx_engagements_ay ON engagements(ay);
CREATE INDEX idx_documents_engagement_id ON documents(engagement_id);
CREATE INDEX idx_comments_engagement_id ON comments(engagement_id);
CREATE INDEX idx_activity_log_entity ON activity_log(entity_type, entity_id);
CREATE INDEX idx_activity_log_actor ON activity_log(actor_user_id);
CREATE INDEX idx_activity_log_created_at ON activity_log(created_at DESC);
CREATE INDEX idx_status_transitions_engagement ON status_transitions(engagement_id);
CREATE INDEX idx_checklist_items_checklist_id ON checklist_items(checklist_id);
CREATE INDEX idx_assignments_engagement_id ON assignments(engagement_id);
CREATE INDEX idx_assignments_user_id ON assignments(user_id);

-- Add triggers for updated_at timestamps
CREATE OR REPLACE FUNCTION update_updated_at_column()
RETURNS TRIGGER AS $$
BEGIN
    NEW.updated_at = NOW();
    RETURN NEW;
END;
$$ language 'plpgsql';

CREATE TRIGGER update_users_updated_at BEFORE UPDATE ON users
    FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_clients_updated_at BEFORE UPDATE ON clients
    FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_engagements_updated_at BEFORE UPDATE ON engagements
    FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_checklist_items_updated_at BEFORE UPDATE ON checklist_items
    FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_computations_updated_at BEFORE UPDATE ON computations
    FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();

CREATE TRIGGER update_efiling_updated_at BEFORE UPDATE ON efiling
    FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();
