#!/bin/bash

# ITR Tracker Startup Script
# This script starts the ITR Tracker application in production mode

set -e  # Exit on any error

echo "🚀 Starting ITR Tracker..."

# Function to check if a command exists
command_exists() {
    command -v "$1" >/dev/null 2>&1
}

# Function to check if a port is in use
port_in_use() {
    lsof -i :$1 >/dev/null 2>&1
}

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Print colored output
print_status() {
    echo -e "${GREEN}✓${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}⚠${NC} $1"
}

print_error() {
    echo -e "${RED}✗${NC} $1"
}

print_info() {
    echo -e "${BLUE}ℹ${NC} $1"
}

# Check prerequisites
print_info "Checking prerequisites..."

if ! command_exists python3; then
    print_error "Python 3 is required but not installed."
    exit 1
fi

if ! command_exists node; then
    print_error "Node.js is required but not installed."
    exit 1
fi

if ! command_exists npm; then
    print_error "npm is required but not installed."
    exit 1
fi

print_status "Prerequisites check passed"

# Check environment variables
print_info "Checking environment configuration..."

if [ ! -f .env ]; then
    print_warning ".env file not found, creating from template..."
    if [ -f .env.example ]; then
        cp .env.example .env
        print_info "Please configure .env file with your settings"
    else
        print_error ".env.example not found. Cannot create .env file."
        exit 1
    fi
fi

# Load environment variables
set -a
source .env
set +a

print_status "Environment configuration loaded"

# Check if port is available
TARGET_PORT=${PORT:-5000}
if port_in_use $TARGET_PORT; then
    print_error "Port $TARGET_PORT is already in use. Please stop the service using this port or change the PORT in .env"
    exit 1
fi

# Install dependencies if needed
if [ ! -d "node_modules" ]; then
    print_info "Installing Node.js dependencies..."
    npm install
    print_status "Node.js dependencies installed"
fi

# Check if Python dependencies are installed
if ! python3 -c "import fastapi" 2>/dev/null; then
    print_info "Installing Python dependencies..."
    pip3 install -r server/requirements.txt
    print_status "Python dependencies installed"
fi

# Run database migrations
print_info "Running database migrations..."
if command_exists make; then
    make migrate
else
    npm run db:push
fi
print_status "Database migrations completed"

# Build frontend if needed
if [ ! -d "dist" ]; then
    print_info "Building frontend assets..."
    npm run build
    print_status "Frontend build completed"
fi

# Create uploads directory
mkdir -p uploads
chmod 755 uploads

# Create logs directory
mkdir -p logs
chmod 755 logs

# Start the application
print_info "Starting ITR Tracker application..."

# Set production environment
export NODE_ENV=production

# Start the server
if command_exists make; then
    make start
else
    npm run start
fi &

SERVER_PID=$!

# Wait a moment for server to start
sleep 3

# Check if server started successfully
if ! kill -0 $SERVER_PID 2>/dev/null; then
    print_error "Failed to start ITR Tracker server"
    exit 1
fi

print_status "ITR Tracker started successfully!"
echo ""
print_info "Application Details:"
echo "  📍 URL: http://localhost:$TARGET_PORT"
echo "  📁 Process ID: $SERVER_PID"
echo "  📊 API Documentation: http://localhost:$TARGET_PORT/docs"
echo ""
print_info "Demo Login Credentials:"
echo "  👑 Partner: admin@ca-firm.com / admin123"
echo "  👨‍💼 Manager: manager@ca-firm.com / manager123" 
echo "  👩‍💻 Staff: staff@ca-firm.com / staff123"
echo "  🧑‍💼 Client: client1@example.com / client123"
echo "  🔍 Auditor: auditor@firm.com / audit123"
echo ""
print_info "Logs:"
echo "  📝 Application logs: logs/app.log"
echo "  📈 Access logs: logs/access.log"
echo ""
print_warning "Press Ctrl+C to stop the server"

# Function to handle shutdown
shutdown() {
    echo ""
    print_info "Shutting down ITR Tracker..."
    kill $SERVER_PID 2>/dev/null || true
    wait $SERVER_PID 2>/dev/null || true
    print_status "ITR Tracker stopped"
    exit 0
}

# Trap interrupt signals
trap shutdown SIGINT SIGTERM

# Wait for the server process
wait $SERVER_PID
