import jwt from 'jsonwebtoken';
import bcrypt from 'bcryptjs';
import type { User } from '@shared/schema';

const JWT_SECRET = process.env.JWT_SECRET || 'your-super-secret-jwt-key';
const JWT_EXPIRES_IN = process.env.JWT_EXPIRES_IN || '7d';
const JWT_REFRESH_EXPIRES_IN = process.env.JWT_REFRESH_EXPIRES_IN || '30d';

export interface JWTPayload {
  userId: string;
  email: string;
  role: string;
  iat?: number;
  exp?: number;
}

export interface AuthTokens {
  accessToken: string;
  refreshToken: string;
}

export class AuthService {
  static async hashPassword(password: string): Promise<string> {
    const saltRounds = 12;
    return await bcrypt.hash(password, saltRounds);
  }

  static async verifyPassword(password: string, hashedPassword: string): Promise<boolean> {
    return await bcrypt.compare(password, hashedPassword);
  }

  static generateTokens(user: User): AuthTokens {
    const payload: JWTPayload = {
      userId: user.id,
      email: user.email,
      role: user.role
    };

    const accessToken = jwt.sign(payload, JWT_SECRET, {
      expiresIn: JWT_EXPIRES_IN
    } as jwt.SignOptions);

    const refreshToken = jwt.sign(payload, JWT_SECRET, {
      expiresIn: JWT_REFRESH_EXPIRES_IN
    } as jwt.SignOptions);

    return { accessToken, refreshToken };
  }

  static verifyToken(token: string): JWTPayload {
    try {
      return jwt.verify(token, JWT_SECRET) as JWTPayload;
    } catch (error) {
      throw new Error('Invalid or expired token');
    }
  }

  static extractTokenFromHeader(authHeader?: string): string | null {
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return null;
    }
    return authHeader.substring(7);
  }
}

// RBAC permissions
export const PERMISSIONS = {
  // Users
  USER_CREATE: 'user:create',
  USER_READ: 'user:read',
  USER_UPDATE: 'user:update',
  USER_DELETE: 'user:delete',

  // Clients
  CLIENT_CREATE: 'client:create',
  CLIENT_READ: 'client:read',
  CLIENT_UPDATE: 'client:update',
  CLIENT_DELETE: 'client:delete',

  // Engagements
  ENGAGEMENT_CREATE: 'engagement:create',
  ENGAGEMENT_READ: 'engagement:read',
  ENGAGEMENT_UPDATE: 'engagement:update',
  ENGAGEMENT_DELETE: 'engagement:delete',
  ENGAGEMENT_TRANSITION: 'engagement:transition',

  // Documents
  DOCUMENT_UPLOAD: 'document:upload',
  DOCUMENT_READ: 'document:read',
  DOCUMENT_DELETE: 'document:delete',

  // Reviews
  REVIEW_MANAGER: 'review:manager',
  REVIEW_PARTNER: 'review:partner',

  // Admin
  ADMIN_FULL: 'admin:full',
  AUDIT_READ: 'audit:read',
  SETTINGS_MANAGE: 'settings:manage'
};

export const ROLE_PERMISSIONS = {
  PARTNER: [
    PERMISSIONS.USER_CREATE,
    PERMISSIONS.USER_READ,
    PERMISSIONS.USER_UPDATE,
    PERMISSIONS.USER_DELETE,
    PERMISSIONS.CLIENT_CREATE,
    PERMISSIONS.CLIENT_READ,
    PERMISSIONS.CLIENT_UPDATE,
    PERMISSIONS.CLIENT_DELETE,
    PERMISSIONS.ENGAGEMENT_CREATE,
    PERMISSIONS.ENGAGEMENT_READ,
    PERMISSIONS.ENGAGEMENT_UPDATE,
    PERMISSIONS.ENGAGEMENT_DELETE,
    PERMISSIONS.ENGAGEMENT_TRANSITION,
    PERMISSIONS.DOCUMENT_UPLOAD,
    PERMISSIONS.DOCUMENT_READ,
    PERMISSIONS.DOCUMENT_DELETE,
    PERMISSIONS.REVIEW_MANAGER,
    PERMISSIONS.REVIEW_PARTNER,
    PERMISSIONS.ADMIN_FULL,
    PERMISSIONS.AUDIT_READ,
    PERMISSIONS.SETTINGS_MANAGE
  ],
  MANAGER: [
    PERMISSIONS.CLIENT_READ,
    PERMISSIONS.CLIENT_UPDATE,
    PERMISSIONS.ENGAGEMENT_CREATE,
    PERMISSIONS.ENGAGEMENT_READ,
    PERMISSIONS.ENGAGEMENT_UPDATE,
    PERMISSIONS.ENGAGEMENT_TRANSITION,
    PERMISSIONS.DOCUMENT_UPLOAD,
    PERMISSIONS.DOCUMENT_READ,
    PERMISSIONS.REVIEW_MANAGER,
    PERMISSIONS.AUDIT_READ
  ],
  STAFF: [
    PERMISSIONS.CLIENT_READ,
    PERMISSIONS.ENGAGEMENT_READ,
    PERMISSIONS.ENGAGEMENT_UPDATE,
    PERMISSIONS.DOCUMENT_UPLOAD,
    PERMISSIONS.DOCUMENT_READ
  ],
  CLIENT: [
    PERMISSIONS.ENGAGEMENT_READ,
    PERMISSIONS.DOCUMENT_READ
  ],
  AUDITOR: [
    PERMISSIONS.ENGAGEMENT_READ,
    PERMISSIONS.DOCUMENT_READ,
    PERMISSIONS.AUDIT_READ
  ]
};

export class RBACService {
  static hasPermission(userRole: string, permission: string): boolean {
    const rolePermissions = ROLE_PERMISSIONS[userRole as keyof typeof ROLE_PERMISSIONS];
    return rolePermissions?.includes(permission) || false;
  }

  static requirePermission(userRole: string, permission: string): void {
    if (!this.hasPermission(userRole, permission)) {
      throw new Error(`Access denied: Missing permission ${permission}`);
    }
  }

  static canAccessEngagement(userRole: string, userId: string, engagement: any): boolean {
    // Partners and managers can access all engagements
    if (userRole === 'PARTNER' || userRole === 'MANAGER') {
      return true;
    }

    // Staff can access engagements they're assigned to
    if (userRole === 'STAFF') {
      // This would need to check assignments table
      return true; // Simplified for now
    }

    // Clients can only access their own engagements
    if (userRole === 'CLIENT') {
      return engagement.clientId === userId;
    }

    // Auditors can access all (read-only)
    if (userRole === 'AUDITOR') {
      return true;
    }

    return false;
  }
}
