import type { Express } from "express";
import { createServer, type Server } from "http";
import multer from 'multer';
import { storage } from "./storage";
import { AuthService } from "./auth";
import { authMiddleware, requirePermission, requireRole, errorHandler, requestLogger, rateLimiter, type AuthenticatedRequest } from "./middleware";
import { ActivityLogService } from "./services/activityLog";
import { FileStorageService } from "./services/fileStorage";
import { WorkflowEngine } from "./services/workflowEngine";
import { ReminderService } from "./services/reminderService";
import { 
  validateSchema, 
  validateQuery,
  createClientSchema,
  createEngagementSchema,
  createCommentSchema,
  updateChecklistItemSchema,
  registerUserSchema,
  loginSchema,
  statusTransitionSchema,
  createComputationSchema,
  searchSchema
} from "./utils/validators";
import { PERMISSIONS } from "./auth";

// Configure multer for file uploads
const upload = multer({
  storage: multer.memoryStorage(),
  limits: {
    fileSize: 10 * 1024 * 1024 // 10MB limit
  }
});

export async function registerRoutes(app: Express): Promise<Server> {
  // Apply global middleware
  app.use(requestLogger);
  // Temporarily disable rate limiter during development
  // app.use(rateLimiter());

  // Health check
  app.get('/api/health', (req, res) => {
    res.json({ status: 'ok', timestamp: new Date().toISOString() });
  });

  // Authentication routes
  app.post('/api/auth/register', validateSchema(registerUserSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const hashedPassword = await AuthService.hashPassword(req.validatedData.password);
      
      const user = await storage.createUser({
        ...req.validatedData,
        password: hashedPassword
      });

      // Remove password from response
      const { password, ...userResponse } = user;
      
      const tokens = AuthService.generateTokens(user);

      await ActivityLogService.logActivity({
        actorUserId: user.id,
        actorRole: user.role,
        action: ActivityLogService.ACTIONS.CREATE,
        entityType: ActivityLogService.ENTITY_TYPES.USER,
        entityId: user.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent'),
        afterData: userResponse
      });

      res.status(201).json({
        user: userResponse,
        ...tokens
      });
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.post('/api/auth/login', validateSchema(loginSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const user = await storage.getUserByEmail(req.validatedData.email);
      
      if (!user || !user.isActive) {
        return res.status(401).json({ message: 'Invalid credentials' });
      }

      const validPassword = await AuthService.verifyPassword(req.validatedData.password, user.password);
      
      if (!validPassword) {
        return res.status(401).json({ message: 'Invalid credentials' });
      }

      // Update last login
      await storage.updateUser(user.id, { lastLoginAt: new Date() });

      const tokens = AuthService.generateTokens(user);
      const { password, ...userResponse } = user;

      await ActivityLogService.logActivity({
        actorUserId: user.id,
        actorRole: user.role,
        action: ActivityLogService.ACTIONS.LOGIN,
        entityType: ActivityLogService.ENTITY_TYPES.USER,
        entityId: user.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent')
      });

      res.json({
        user: userResponse,
        ...tokens
      });
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.get('/api/auth/me', authMiddleware, async (req: AuthenticatedRequest, res) => {
    try {
      const user = await storage.getUser(req.user!.id);
      if (!user) {
        return res.status(404).json({ message: 'User not found' });
      }

      const { password, ...userResponse } = user;
      res.json(userResponse);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Users routes
  app.get('/api/users', authMiddleware, requirePermission(PERMISSIONS.USER_READ), async (req: AuthenticatedRequest, res) => {
    try {
      const users = await storage.getUsers();
      const sanitizedUsers = users.map(({ password, ...user }) => user);
      res.json(sanitizedUsers);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Clients routes
  app.get('/api/clients', authMiddleware, requirePermission(PERMISSIONS.CLIENT_READ), validateQuery(searchSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const result = await storage.getClients(req.validatedQuery);
      res.json(result);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.get('/api/clients/:id', authMiddleware, requirePermission(PERMISSIONS.CLIENT_READ), async (req: AuthenticatedRequest, res) => {
    try {
      const client = await storage.getClient(req.params.id);
      if (!client) {
        return res.status(404).json({ message: 'Client not found' });
      }
      res.json(client);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.post('/api/clients', authMiddleware, requirePermission(PERMISSIONS.CLIENT_CREATE), validateSchema(createClientSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const client = await storage.createClient(req.validatedData);

      await ActivityLogService.logActivity({
        actorUserId: req.user!.id,
        actorRole: req.user!.role,
        action: ActivityLogService.ACTIONS.CREATE,
        entityType: ActivityLogService.ENTITY_TYPES.CLIENT,
        entityId: client.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent'),
        afterData: client
      });

      res.status(201).json(client);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.put('/api/clients/:id', authMiddleware, requirePermission(PERMISSIONS.CLIENT_UPDATE), validateSchema(createClientSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const existingClient = await storage.getClient(req.params.id);
      if (!existingClient) {
        return res.status(404).json({ message: 'Client not found' });
      }

      const updatedClient = await storage.updateClient(req.params.id, req.validatedData);

      await ActivityLogService.logActivity({
        actorUserId: req.user!.id,
        actorRole: req.user!.role,
        action: ActivityLogService.ACTIONS.UPDATE,
        entityType: ActivityLogService.ENTITY_TYPES.CLIENT,
        entityId: req.params.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent'),
        beforeData: existingClient,
        afterData: updatedClient
      });

      res.json(updatedClient);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Engagements routes
  app.get('/api/engagements', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_READ), validateQuery(searchSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const result = await storage.getEngagements(req.validatedQuery);
      res.json(result);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.get('/api/engagements/:id', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_READ), async (req: AuthenticatedRequest, res) => {
    try {
      const engagement = await storage.getEngagement(req.params.id);
      if (!engagement) {
        return res.status(404).json({ message: 'Engagement not found' });
      }
      res.json(engagement);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.post('/api/engagements', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_CREATE), validateSchema(createEngagementSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const engagement = await storage.createEngagement(req.validatedData);

      // Auto-create checklist for the engagement
      await WorkflowEngine.autoCreateChecklist(engagement.id, engagement.itrFormType);

      await ActivityLogService.logActivity({
        actorUserId: req.user!.id,
        actorRole: req.user!.role,
        action: ActivityLogService.ACTIONS.CREATE,
        entityType: ActivityLogService.ENTITY_TYPES.ENGAGEMENT,
        entityId: engagement.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent'),
        afterData: engagement
      });

      res.status(201).json(engagement);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.post('/api/engagements/:id/transition', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_TRANSITION), validateSchema(statusTransitionSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const engagement = await WorkflowEngine.transitionStatus(
        req.params.id,
        req.validatedData.status,
        req.user!.id,
        req.user!.role,
        req.validatedData.reason,
        req.ip
      );

      res.json(engagement);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Documents routes
  app.get('/api/engagements/:id/documents', authMiddleware, requirePermission(PERMISSIONS.DOCUMENT_READ), async (req: AuthenticatedRequest, res) => {
    try {
      const documents = await storage.getDocumentsByEngagement(req.params.id);
      res.json(documents);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.post('/api/engagements/:id/documents', authMiddleware, requirePermission(PERMISSIONS.DOCUMENT_UPLOAD), upload.single('file'), async (req: AuthenticatedRequest, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ message: 'No file uploaded' });
      }

      const docType = req.body.docType || 'GENERAL';
      
      const uploadedFile = await FileStorageService.uploadFile(
        req.file,
        req.params.id,
        req.user!.id,
        docType
      );

      await ActivityLogService.logActivity({
        actorUserId: req.user!.id,
        actorRole: req.user!.role,
        action: ActivityLogService.ACTIONS.UPLOAD_DOCUMENT,
        entityType: ActivityLogService.ENTITY_TYPES.DOCUMENT,
        entityId: req.params.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent'),
        afterData: uploadedFile
      });

      res.status(201).json(uploadedFile);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.get('/api/documents/:id/download', authMiddleware, requirePermission(PERMISSIONS.DOCUMENT_READ), async (req: AuthenticatedRequest, res) => {
    try {
      const { filePath, fileName, mimeType } = await FileStorageService.downloadFile(req.params.id, req.user!.id);

      await ActivityLogService.logActivity({
        actorUserId: req.user!.id,
        actorRole: req.user!.role,
        action: ActivityLogService.ACTIONS.DOWNLOAD_DOCUMENT,
        entityType: ActivityLogService.ENTITY_TYPES.DOCUMENT,
        entityId: req.params.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent')
      });

      res.setHeader('Content-Type', mimeType);
      res.setHeader('Content-Disposition', `attachment; filename="${fileName}"`);
      res.sendFile(filePath);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Comments routes
  app.get('/api/engagements/:id/comments', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_READ), async (req: AuthenticatedRequest, res) => {
    try {
      const comments = await storage.getCommentsByEngagement(req.params.id);
      res.json(comments);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.post('/api/comments', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_READ), validateSchema(createCommentSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const commentData = {
        ...req.validatedData,
        authorId: req.user!.id
      };

      const comment = await storage.createComment(commentData);

      await ActivityLogService.logActivity({
        actorUserId: req.user!.id,
        actorRole: req.user!.role,
        action: ActivityLogService.ACTIONS.ADD_COMMENT,
        entityType: ActivityLogService.ENTITY_TYPES.COMMENT,
        entityId: comment.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent'),
        afterData: comment
      });

      res.status(201).json(comment);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Checklist routes
  app.get('/api/engagements/:id/checklist', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_READ), async (req: AuthenticatedRequest, res) => {
    try {
      const items = await storage.getChecklistItemsByEngagement(req.params.id);
      res.json(items);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.put('/api/checklist-items/:id', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_UPDATE), validateSchema(updateChecklistItemSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const updatedItem = await storage.updateChecklistItem(req.params.id, req.validatedData);

      await ActivityLogService.logActivity({
        actorUserId: req.user!.id,
        actorRole: req.user!.role,
        action: ActivityLogService.ACTIONS.UPDATE,
        entityType: ActivityLogService.ENTITY_TYPES.CHECKLIST_ITEM,
        entityId: req.params.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent'),
        afterData: updatedItem
      });

      res.json(updatedItem);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Computations routes
  app.get('/api/engagements/:id/computations', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_READ), async (req: AuthenticatedRequest, res) => {
    try {
      const computations = await storage.getComputationsByEngagement(req.params.id);
      res.json(computations);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.post('/api/computations', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_UPDATE), validateSchema(createComputationSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const computationData = {
        ...req.validatedData,
        preparedBy: req.user!.id
      };

      const computation = await storage.createComputation(computationData);

      await ActivityLogService.logActivity({
        actorUserId: req.user!.id,
        actorRole: req.user!.role,
        action: ActivityLogService.ACTIONS.CREATE,
        entityType: ActivityLogService.ENTITY_TYPES.COMPUTATION,
        entityId: computation.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent'),
        afterData: computation
      });

      res.status(201).json(computation);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Dashboard routes
  app.get('/api/dashboard/stats', authMiddleware, async (req: AuthenticatedRequest, res) => {
    try {
      const stats = await storage.getDashboardStats(req.user!.id);
      res.json(stats);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.get('/api/dashboard/pipeline', authMiddleware, async (req: AuthenticatedRequest, res) => {
    try {
      const pipeline = await storage.getPipelineStats();
      res.json(pipeline);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.get('/api/dashboard/activity', authMiddleware, async (req: AuthenticatedRequest, res) => {
    try {
      const activity = await storage.getRecentActivity(10);
      res.json(activity);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Activity log routes
  app.get('/api/activity-log', authMiddleware, requirePermission(PERMISSIONS.AUDIT_READ), validateQuery(searchSchema), async (req: AuthenticatedRequest, res) => {
    try {
      const result = await ActivityLogService.getActivityLog(req.validatedQuery);
      res.json(result);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  app.get('/api/activity-log/export', authMiddleware, requirePermission(PERMISSIONS.AUDIT_READ), async (req: AuthenticatedRequest, res) => {
    try {
      const format = req.query.format as 'csv' | 'pdf' || 'csv';
      const exportData = await ActivityLogService.exportActivityLog(format);

      await ActivityLogService.logActivity({
        actorUserId: req.user!.id,
        actorRole: req.user!.role,
        action: ActivityLogService.ACTIONS.EXPORT_DATA,
        entityType: 'ACTIVITY_LOG',
        entityId: 'export',
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent'),
        afterData: { format }
      });

      res.setHeader('Content-Type', format === 'csv' ? 'text/csv' : 'application/pdf');
      res.setHeader('Content-Disposition', `attachment; filename="activity-log.${format}"`);
      res.send(exportData);
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Reminder routes
  app.get('/api/reminder-templates', authMiddleware, requirePermission(PERMISSIONS.SETTINGS_MANAGE), (req, res) => {
    const templates = ReminderService.getTemplates();
    res.json(templates);
  });

  app.post('/api/reminders/send', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_UPDATE), async (req: AuthenticatedRequest, res) => {
    try {
      const { engagementId, templateKey, payload, channel } = req.body;
      
      const success = await ReminderService.sendReminder(engagementId, templateKey, payload, channel);
      
      res.json({ success });
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Evidence pack export
  app.get('/api/engagements/:id/evidence-pack', authMiddleware, requirePermission(PERMISSIONS.ENGAGEMENT_READ), async (req: AuthenticatedRequest, res) => {
    try {
      const evidencePackPath = await FileStorageService.createEvidencePack(req.params.id);

      await ActivityLogService.logActivity({
        actorUserId: req.user!.id,
        actorRole: req.user!.role,
        action: ActivityLogService.ACTIONS.EXPORT_DATA,
        entityType: ActivityLogService.ENTITY_TYPES.ENGAGEMENT,
        entityId: req.params.id,
        ip: req.ip || 'unknown',
        userAgent: req.get('User-Agent'),
        afterData: { type: 'evidence_pack' }
      });

      res.json({ downloadUrl: evidencePackPath });
    } catch (error) {
      res.status(400).json({ message: error instanceof Error ? error.message : 'An error occurred' });
    }
  });

  // Apply error handler
  app.use(errorHandler);

  const httpServer = createServer(app);
  return httpServer;
}
