#!/usr/bin/env python3
"""
Seed script for ITR Tracker application.
Creates demo data for development and testing.
"""

import asyncio
import sys
import os
from datetime import datetime, timedelta
from typing import List

# Add the project root to Python path
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from server.storage import DatabaseStorage
from server.auth import AuthService
from server.services.activityLog import ActivityLogService
import uuid

class DataSeeder:
    def __init__(self):
        self.storage = DatabaseStorage()

    async def seed_users(self) -> List[str]:
        """Create demo users with different roles."""
        print("Creating demo users...")
        
        users_data = [
            {
                "name": "Amit Mehta",
                "email": "admin@ca-firm.com",
                "mobile": "9876543210",
                "password": await AuthService.hashPassword("admin123"),
                "role": "PARTNER",
                "isActive": True
            },
            {
                "name": "Priya Sharma",
                "email": "manager@ca-firm.com",
                "mobile": "9876543211",
                "password": await AuthService.hashPassword("manager123"),
                "role": "MANAGER",
                "isActive": True
            },
            {
                "name": "Ankit Patel",
                "email": "staff@ca-firm.com",
                "mobile": "9876543212",
                "password": await AuthService.hashPassword("staff123"),
                "role": "STAFF",
                "isActive": True
            },
            {
                "name": "Rohit Singh",
                "email": "staff2@ca-firm.com",
                "mobile": "9876543213",
                "password": await AuthService.hashPassword("staff123"),
                "role": "STAFF",
                "isActive": True
            },
            {
                "name": "Kavya Nair",
                "email": "staff3@ca-firm.com",
                "mobile": "9876543214",
                "password": await AuthService.hashPassword("staff123"),
                "role": "STAFF",
                "isActive": True
            },
            {
                "name": "Rajesh Kumar",
                "email": "client1@example.com",
                "mobile": "9876543215",
                "password": await AuthService.hashPassword("client123"),
                "role": "CLIENT",
                "isActive": True
            },
            {
                "name": "Audit Officer",
                "email": "auditor@firm.com",
                "mobile": "9876543216",
                "password": await AuthService.hashPassword("audit123"),
                "role": "AUDITOR",
                "isActive": True
            }
        ]

        user_ids = []
        for user_data in users_data:
            try:
                user = await self.storage.createUser(user_data)
                user_ids.append(user.id)
                print(f"✓ Created user: {user.name} ({user.email})")
            except Exception as e:
                print(f"✗ Failed to create user {user_data['email']}: {e}")

        return user_ids

    async def seed_clients(self) -> List[str]:
        """Create demo clients."""
        print("\nCreating demo clients...")
        
        clients_data = [
            {
                "displayName": "Rajesh Kumar",
                "legalName": "Rajesh Kumar",
                "pan": "ABCDE1234F",
                "aadhaarLast4": "7892",
                "dateOfBirth": datetime(1985, 5, 15),
                "address": {
                    "line1": "123 MG Road",
                    "city": "Mumbai",
                    "state": "Maharashtra",
                    "pincode": "400001",
                    "country": "India"
                },
                "contacts": [{"type": "primary", "email": "rajesh@example.com", "mobile": "9876543210"}],
                "riskCategory": "MEDIUM"
            },
            {
                "displayName": "Sunita Agarwal",
                "legalName": "Sunita Agarwal", 
                "pan": "SUNAG4567A",
                "aadhaarLast4": "4567",
                "dateOfBirth": datetime(1978, 12, 3),
                "address": {
                    "line1": "456 Park Street",
                    "city": "Delhi",
                    "state": "Delhi",
                    "pincode": "110001",
                    "country": "India"
                },
                "contacts": [{"type": "primary", "email": "sunita@example.com", "mobile": "9876543220"}],
                "riskCategory": "LOW"
            },
            {
                "displayName": "TechCorp Pvt Ltd",
                "legalName": "TechCorp Private Limited",
                "pan": "TCORP8901C",
                "incorporationDate": datetime(2015, 3, 20),
                "address": {
                    "line1": "Tower A, Business Park",
                    "city": "Bangalore",
                    "state": "Karnataka", 
                    "pincode": "560001",
                    "country": "India"
                },
                "contacts": [{"type": "business", "email": "accounts@techcorp.com", "mobile": "9876543230"}],
                "riskCategory": "HIGH"
            },
            {
                "displayName": "Vishnu Enterprises",
                "legalName": "Vishnu Enterprises",
                "pan": "VISHN3456E",
                "incorporationDate": datetime(2010, 8, 10),
                "address": {
                    "line1": "Industrial Area Phase 1",
                    "city": "Pune",
                    "state": "Maharashtra",
                    "pincode": "411001", 
                    "country": "India"
                },
                "contacts": [{"type": "business", "email": "vishnu@enterprises.com", "mobile": "9876543240"}],
                "riskCategory": "MEDIUM"
            },
            {
                "displayName": "Global Traders LLP",
                "legalName": "Global Traders Limited Liability Partnership",
                "pan": "GLOBL2345T",
                "incorporationDate": datetime(2018, 1, 5),
                "address": {
                    "line1": "Export House Complex",
                    "city": "Chennai",
                    "state": "Tamil Nadu",
                    "pincode": "600001",
                    "country": "India"
                },
                "contacts": [{"type": "business", "email": "trade@global.com", "mobile": "9876543250"}],
                "riskCategory": "HIGH"
            }
        ]

        client_ids = []
        for client_data in clients_data:
            try:
                client = await self.storage.createClient(client_data)
                client_ids.append(client.id)
                print(f"✓ Created client: {client.displayName}")
            except Exception as e:
                print(f"✗ Failed to create client {client_data['displayName']}: {e}")

        return client_ids

    async def seed_engagements(self, client_ids: List[str]) -> List[str]:
        """Create demo engagements."""
        print("\nCreating demo engagements...")
        
        engagements_data = [
            {
                "clientId": client_ids[0],
                "ay": "2024-25",
                "itrFormType": "ITR-2",
                "tier": "INDIVIDUAL",
                "status": "COMPUTATION_PREPARED",
                "isAuditApplicable": False,
                "priority": "HIGH",
                "dueDates": {"filingDue": datetime(2024, 7, 31).isoformat()}
            },
            {
                "clientId": client_ids[1], 
                "ay": "2024-25",
                "itrFormType": "ITR-1",
                "tier": "INDIVIDUAL",
                "status": "MANAGER_REVIEW",
                "isAuditApplicable": False,
                "priority": "MEDIUM",
                "dueDates": {"filingDue": datetime(2024, 7, 31).isoformat()}
            },
            {
                "clientId": client_ids[2],
                "ay": "2024-25", 
                "itrFormType": "ITR-6",
                "tier": "COMPANY",
                "status": "ITR_FILED_ACK_CAPTURED",
                "isAuditApplicable": True,
                "priority": "LOW",
                "dueDates": {"filingDue": datetime(2024, 10, 31).isoformat()}
            },
            {
                "clientId": client_ids[3],
                "ay": "2024-25",
                "itrFormType": "ITR-4", 
                "tier": "FIRM",
                "status": "DATA_RECEIVED_PRELIMINARY",
                "isAuditApplicable": False,
                "priority": "MEDIUM",
                "dueDates": {"filingDue": datetime(2024, 7, 31).isoformat()}
            },
            {
                "clientId": client_ids[4],
                "ay": "2024-25",
                "itrFormType": "ITR-5",
                "tier": "FIRM", 
                "status": "PARTNER_REVIEW",
                "isAuditApplicable": True,
                "priority": "CRITICAL",
                "dueDates": {"filingDue": datetime(2024, 10, 31).isoformat()}
            }
        ]

        engagement_ids = []
        for engagement_data in engagements_data:
            try:
                engagement = await self.storage.createEngagement(engagement_data)
                engagement_ids.append(engagement.id)
                print(f"✓ Created engagement: {engagement.ay} - {engagement.itrFormType}")
            except Exception as e:
                print(f"✗ Failed to create engagement: {e}")

        return engagement_ids

    async def seed_activity_log(self, user_ids: List[str], engagement_ids: List[str]):
        """Create demo activity log entries."""
        print("\nCreating demo activity log entries...")
        
        activities = [
            {
                "actorUserId": user_ids[1],  # Priya Sharma
                "actorRole": "MANAGER",
                "action": "APPROVE_REVIEW",
                "entityType": "ENGAGEMENT", 
                "entityId": engagement_ids[0],
                "ip": "192.168.1.45"
            },
            {
                "actorUserId": user_ids[2],  # Ankit Patel
                "actorRole": "STAFF",
                "action": "UPLOAD_DOCUMENT",
                "entityType": "DOCUMENT",
                "entityId": str(uuid.uuid4()),
                "ip": "192.168.1.42"
            },
            {
                "actorUserId": None,
                "actorRole": "SYSTEM", 
                "action": "SLA_BREACH",
                "entityType": "ENGAGEMENT",
                "entityId": engagement_ids[1],
                "ip": "127.0.0.1"
            },
            {
                "actorUserId": user_ids[3],  # Rohit Singh
                "actorRole": "STAFF",
                "action": "ADD_COMMENT", 
                "entityType": "ENGAGEMENT",
                "entityId": engagement_ids[2],
                "ip": "192.168.1.43"
            }
        ]

        for activity_data in activities:
            try:
                await ActivityLogService.logActivity(activity_data)
                print(f"✓ Created activity log: {activity_data['action']}")
            except Exception as e:
                print(f"✗ Failed to create activity log: {e}")

    async def run(self):
        """Run the complete seeding process."""
        print("🌱 Starting ITR Tracker data seeding...\n")
        
        try:
            # Seed users
            user_ids = await self.seed_users()
            
            # Seed clients
            client_ids = await self.seed_clients()
            
            # Seed engagements
            engagement_ids = await self.seed_engagements(client_ids)
            
            # Seed activity log
            await self.seed_activity_log(user_ids, engagement_ids)
            
            print(f"\n✅ Seeding completed successfully!")
            print(f"📊 Created: {len(user_ids)} users, {len(client_ids)} clients, {len(engagement_ids)} engagements")
            print("\n🔑 Demo Login Credentials:")
            print("Partner: admin@ca-firm.com / admin123")
            print("Manager: manager@ca-firm.com / manager123") 
            print("Staff: staff@ca-firm.com / staff123")
            print("Client: client1@example.com / client123")
            print("Auditor: auditor@firm.com / audit123")
            
        except Exception as e:
            print(f"\n❌ Seeding failed: {e}")
            sys.exit(1)

if __name__ == "__main__":
    seeder = DataSeeder()
    asyncio.run(seeder.run())
