import { storage } from '../storage';
import { createHash, createHmac } from 'crypto';
import type { ActivityLogEntry } from '@shared/schema';

const HMAC_SECRET = process.env.HMAC_SECRET || 'your-hmac-secret-key';

export interface LogActivity {
  actorUserId?: string;
  actorRole: string;
  action: string;
  entityType: string;
  entityId: string;
  ip: string;
  userAgent?: string;
  beforeData?: any;
  afterData?: any;
}

export class ActivityLogService {
  static async logActivity(activity: LogActivity): Promise<ActivityLogEntry> {
    const beforeJson = activity.beforeData ? JSON.stringify(activity.beforeData) : null;
    const afterJson = activity.afterData ? JSON.stringify(activity.afterData) : null;

    // Create signature hash for tamper detection
    const signatureData = {
      actorUserId: activity.actorUserId,
      actorRole: activity.actorRole,
      action: activity.action,
      entityType: activity.entityType,
      entityId: activity.entityId,
      ip: activity.ip,
      userAgent: activity.userAgent,
      beforeJson,
      afterJson,
      timestamp: new Date().toISOString()
    };

    const signatureHash = this.generateSignature(signatureData);

    const logEntry = await storage.createActivityLog({
      actorUserId: activity.actorUserId || null,
      actorRole: activity.actorRole,
      action: activity.action,
      entityType: activity.entityType,
      entityId: activity.entityId,
      ip: activity.ip,
      userAgent: activity.userAgent || null,
      beforeJson: beforeJson ? JSON.parse(beforeJson) : null,
      afterJson: afterJson ? JSON.parse(afterJson) : null,
      signatureHash
    });

    return logEntry;
  }

  static generateSignature(data: any): string {
    const dataString = JSON.stringify(data, Object.keys(data).sort());
    return createHmac('sha256', HMAC_SECRET).update(dataString).digest('hex');
  }

  static verifySignature(logEntry: ActivityLogEntry): boolean {
    const signatureData = {
      actorUserId: logEntry.actorUserId,
      actorRole: logEntry.actorRole,
      action: logEntry.action,
      entityType: logEntry.entityType,
      entityId: logEntry.entityId,
      ip: logEntry.ip,
      userAgent: logEntry.userAgent,
      beforeJson: logEntry.beforeJson ? JSON.stringify(logEntry.beforeJson) : null,
      afterJson: logEntry.afterJson ? JSON.stringify(logEntry.afterJson) : null,
      timestamp: logEntry.createdAt?.toISOString()
    };

    const expectedSignature = this.generateSignature(signatureData);
    return expectedSignature === logEntry.signatureHash;
  }

  static async getActivityLog(filters?: {
    engagementId?: string;
    entityType?: string;
    limit?: number;
    offset?: number;
  }) {
    return await storage.getActivityLog(filters);
  }

  static async exportActivityLog(format: 'csv' | 'pdf' = 'csv', filters?: any): Promise<string> {
    const { logs } = await storage.getActivityLog(filters);
    
    if (format === 'csv') {
      return this.generateCSV(logs);
    } else {
      return this.generatePDF(logs);
    }
  }

  private static generateCSV(logs: ActivityLogEntry[]): string {
    const headers = [
      'Timestamp',
      'Actor User ID',
      'Actor Role', 
      'Action',
      'Entity Type',
      'Entity ID',
      'IP Address',
      'User Agent',
      'Signature Hash'
    ];

    const rows = logs.map(log => [
      log.createdAt?.toISOString() || '',
      log.actorUserId || '',
      log.actorRole,
      log.action,
      log.entityType,
      log.entityId,
      log.ip,
      log.userAgent || '',
      log.signatureHash
    ]);

    return [headers, ...rows].map(row => 
      row.map(cell => `"${cell}"`).join(',')
    ).join('\n');
  }

  private static generatePDF(logs: ActivityLogEntry[]): string {
    // This would generate a PDF using a library like puppeteer or jsPDF
    // For now, return a placeholder
    return 'PDF generation would be implemented here';
  }

  // Common activity types
  static readonly ACTIONS = {
    // Authentication
    LOGIN: 'LOGIN',
    LOGOUT: 'LOGOUT',
    
    // CRUD operations
    CREATE: 'CREATE',
    UPDATE: 'UPDATE',
    DELETE: 'DELETE',
    READ: 'READ',
    
    // Engagement workflow
    STATUS_CHANGE: 'STATUS_CHANGE',
    ASSIGN: 'ASSIGN',
    UNASSIGN: 'UNASSIGN',
    
    // Reviews
    APPROVE_REVIEW: 'APPROVE_REVIEW',
    REJECT_REVIEW: 'REJECT_REVIEW',
    REQUEST_CHANGES: 'REQUEST_CHANGES',
    
    // Documents
    UPLOAD_DOCUMENT: 'UPLOAD_DOCUMENT',
    DOWNLOAD_DOCUMENT: 'DOWNLOAD_DOCUMENT',
    DELETE_DOCUMENT: 'DELETE_DOCUMENT',
    
    // Comments
    ADD_COMMENT: 'ADD_COMMENT',
    MENTION: 'MENTION',
    
    // E-filing
    FILE_ITR: 'FILE_ITR',
    E_VERIFY: 'E_VERIFY',
    
    // System events
    SLA_BREACH: 'SLA_BREACH',
    REMINDER_SENT: 'REMINDER_SENT',
    
    // Admin
    EXPORT_DATA: 'EXPORT_DATA',
    IMPORT_DATA: 'IMPORT_DATA'
  };

  static readonly ENTITY_TYPES = {
    USER: 'USER',
    CLIENT: 'CLIENT',
    ENGAGEMENT: 'ENGAGEMENT',
    DOCUMENT: 'DOCUMENT',
    COMMENT: 'COMMENT',
    CHECKLIST_ITEM: 'CHECKLIST_ITEM',
    COMPUTATION: 'COMPUTATION',
    E_FILING: 'E_FILING'
  };
}
