import fs from 'fs/promises';
import path from 'path';
import { createHash } from 'crypto';
import { storage } from '../storage';
import type { InsertDocument } from '@shared/schema';

const UPLOAD_DIR = process.env.FILE_STORAGE_DIR || path.join(process.cwd(), 'uploads');
const MAX_FILE_SIZE = parseInt(process.env.MAX_FILE_SIZE || '10485760'); // 10MB default
const ALLOWED_MIME_TYPES = [
  'application/pdf',
  'application/vnd.ms-excel',
  'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
  'text/csv',
  'image/jpeg',
  'image/png',
  'image/jpg'
];

export interface UploadedFile {
  originalName: string;
  fileName: string;
  filePath: string;
  mimeType: string;
  size: number;
  checksum: string;
}

export class FileStorageService {
  static async ensureUploadDir(): Promise<void> {
    try {
      await fs.access(UPLOAD_DIR);
    } catch {
      await fs.mkdir(UPLOAD_DIR, { recursive: true });
    }
  }

  static async uploadFile(
    file: Express.Multer.File,
    engagementId: string,
    uploadedBy: string,
    docType: string
  ): Promise<UploadedFile> {
    await this.ensureUploadDir();

    // Validate file
    this.validateFile(file);

    // Generate unique filename
    const timestamp = Date.now();
    const ext = path.extname(file.originalname);
    const fileName = `${timestamp}-${Math.random().toString(36).substring(7)}${ext}`;
    const filePath = path.join(UPLOAD_DIR, fileName);

    // Calculate checksum
    const checksum = this.calculateChecksum(file.buffer);

    // Check for duplicate by checksum
    const existingDocs = await storage.getDocumentsByEngagement(engagementId);
    const duplicate = existingDocs.find(doc => doc.checksum === checksum);
    
    if (duplicate) {
      throw new Error('File already exists (duplicate checksum detected)');
    }

    // Save file to disk
    await fs.writeFile(filePath, file.buffer);

    // Save to database
    const documentData: InsertDocument = {
      engagementId,
      docType,
      fileName: file.originalname,
      filePath: fileName, // Store relative path
      fileSize: file.size,
      checksum,
      uploadedBy,
      clientVisible: false,
      version: 1
    };

    const document = await storage.createDocument(documentData);

    return {
      originalName: file.originalname,
      fileName,
      filePath: fileName,
      mimeType: file.mimetype,
      size: file.size,
      checksum
    };
  }

  static async downloadFile(documentId: string, userId: string): Promise<{
    filePath: string;
    fileName: string;
    mimeType: string;
  }> {
    const document = await storage.getDocument(documentId);
    
    if (!document) {
      throw new Error('Document not found');
    }

    // TODO: Check user permissions for document access

    const fullPath = path.join(UPLOAD_DIR, document.filePath);
    
    try {
      await fs.access(fullPath);
    } catch {
      throw new Error('File not found on disk');
    }

    // Determine MIME type from extension
    const ext = path.extname(document.fileName).toLowerCase();
    const mimeType = this.getMimeType(ext);

    return {
      filePath: fullPath,
      fileName: document.fileName,
      mimeType
    };
  }

  static async deleteFile(documentId: string, userId: string): Promise<void> {
    const document = await storage.getDocument(documentId);
    
    if (!document) {
      throw new Error('Document not found');
    }

    // TODO: Check user permissions for document deletion

    const fullPath = path.join(UPLOAD_DIR, document.filePath);
    
    try {
      await fs.unlink(fullPath);
    } catch (error) {
      console.error('Failed to delete file from disk:', error);
      // Continue with database deletion even if file deletion fails
    }

    await storage.deleteDocument(documentId);
  }

  static validateFile(file: Express.Multer.File): void {
    // Check file size
    if (file.size > MAX_FILE_SIZE) {
      throw new Error(`File size exceeds maximum allowed size of ${MAX_FILE_SIZE} bytes`);
    }

    // Check MIME type
    if (!ALLOWED_MIME_TYPES.includes(file.mimetype)) {
      throw new Error(`File type ${file.mimetype} is not allowed`);
    }

    // Additional security checks
    const ext = path.extname(file.originalname).toLowerCase();
    const allowedExtensions = ['.pdf', '.xlsx', '.xls', '.csv', '.jpg', '.jpeg', '.png'];
    
    if (!allowedExtensions.includes(ext)) {
      throw new Error(`File extension ${ext} is not allowed`);
    }
  }

  static calculateChecksum(buffer: Buffer): string {
    return createHash('sha256').update(buffer).digest('hex');
  }

  static getMimeType(extension: string): string {
    const mimeTypes: Record<string, string> = {
      '.pdf': 'application/pdf',
      '.xlsx': 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
      '.xls': 'application/vnd.ms-excel',
      '.csv': 'text/csv',
      '.jpg': 'image/jpeg',
      '.jpeg': 'image/jpeg',
      '.png': 'image/png'
    };

    return mimeTypes[extension] || 'application/octet-stream';
  }

  static async generateSignedUrl(documentId: string, expiresIn: number = 3600): Promise<string> {
    // For local file storage, we'll generate a temporary access token
    // In production, this would integrate with cloud storage signed URLs
    
    const token = createHash('sha256')
      .update(`${documentId}:${Date.now() + expiresIn * 1000}`)
      .digest('hex');

    return `/api/documents/${documentId}/download?token=${token}`;
  }

  static async createEvidencePack(engagementId: string): Promise<string> {
    // This would create a ZIP file containing all documents for an engagement
    // plus computation records, activity logs, etc.
    
    const documents = await storage.getDocumentsByEngagement(engagementId);
    const computations = await storage.getComputationsByEngagement(engagementId);
    const { logs } = await storage.getActivityLog({ engagementId });

    // TODO: Implement ZIP creation logic
    // For now, return a placeholder path
    return 'evidence-pack-placeholder.zip';
  }
}
