import { storage } from '../storage';
import { ActivityLogService } from './activityLog';
import type { Engagement, User, Client } from '@shared/schema';

export interface ReminderTemplate {
  key: string;
  name: string;
  subject: string;
  body: string;
  channel: 'EMAIL' | 'WHATSAPP' | 'SMS';
}

export interface ReminderPayload {
  clientName: string;
  ay: string;
  dueDate?: string;
  firmName: string;
  link?: string;
  hoursRemaining?: number;
  filedAt?: string;
}

export class ReminderService {
  private static readonly TEMPLATES: ReminderTemplate[] = [
    {
      key: 'DATA_REQUEST',
      name: 'Data Request (Client)',
      subject: 'Document Upload Required - AY {{ay}}',
      body: 'Dear {{clientName}}, please upload AY {{ay}} documents via your secure portal link {{link}} by {{dueDate}}. Regards, {{firmName}}.',
      channel: 'EMAIL'
    },
    {
      key: 'REVIEW_DUE',
      name: 'Review Due (Manager)',
      subject: 'Review Required - {{clientName}}',
      body: 'Engagement {{clientName}}-{{ay}} awaits Manager review since {{date}}; SLA breaches in {{hoursRemaining}}h.',
      channel: 'EMAIL'
    },
    {
      key: 'SIGNOFF_REQUEST',
      name: 'Client Sign-off Request',
      subject: 'ITR Sign-off Required - AY {{ay}}',
      body: 'Dear {{clientName}}, your ITR for AY {{ay}} is ready for review and sign-off. Please check your portal: {{link}}',
      channel: 'EMAIL'
    },
    {
      key: 'E_VERIFY_REMINDER',
      name: 'E-Verification Reminder',
      subject: 'E-Verification Pending - AY {{ay}}',
      body: 'ITR for {{clientName}}-{{ay}} filed {{filedAt}}; e-verification pending. Use Aadhaar/Bank/DSC within 120 hours.',
      channel: 'EMAIL'
    },
    {
      key: 'SLA_BREACH',
      name: 'SLA Breach Alert',
      subject: 'SLA Breach Alert - {{clientName}}',
      body: 'ALERT: Engagement {{clientName}}-{{ay}} has breached SLA targets. Immediate attention required.',
      channel: 'EMAIL'
    },
    {
      key: 'CPC_UPDATE',
      name: 'CPC Status Update',
      subject: 'CPC Processing Update - AY {{ay}}',
      body: 'Dear {{clientName}}, CPC processing update for your ITR AY {{ay}}. Current status: {{status}}.',
      channel: 'EMAIL'
    }
  ];

  static async scheduleReminder(
    engagementId: string,
    templateKey: string,
    scheduleAt: Date,
    payload: ReminderPayload,
    channel: 'EMAIL' | 'WHATSAPP' | 'SMS' = 'EMAIL'
  ): Promise<void> {
    const template = this.TEMPLATES.find(t => t.key === templateKey);
    if (!template) {
      throw new Error(`Template not found: ${templateKey}`);
    }

    // Store reminder in database (would be picked up by scheduler)
    // For now, we'll just log it
    console.log(`Reminder scheduled: ${templateKey} for engagement ${engagementId} at ${scheduleAt}`);
  }

  static async sendReminder(
    engagementId: string,
    templateKey: string,
    payload: ReminderPayload,
    channel: 'EMAIL' | 'WHATSAPP' | 'SMS' = 'EMAIL'
  ): Promise<boolean> {
    const template = this.TEMPLATES.find(t => t.key === templateKey);
    if (!template) {
      throw new Error(`Template not found: ${templateKey}`);
    }

    const message = this.renderTemplate(template, payload);

    try {
      let success = false;

      switch (channel) {
        case 'EMAIL':
          success = await this.sendEmail(payload.clientName, message.subject, message.body);
          break;
        case 'WHATSAPP':
          success = await this.sendWhatsApp(payload.clientName, message.body);
          break;
        case 'SMS':
          success = await this.sendSMS(payload.clientName, message.body);
          break;
      }

      // Log the reminder activity
      await ActivityLogService.logActivity({
        actorRole: 'SYSTEM',
        action: ActivityLogService.ACTIONS.REMINDER_SENT,
        entityType: ActivityLogService.ENTITY_TYPES.ENGAGEMENT,
        entityId: engagementId,
        ip: '127.0.0.1',
        afterData: {
          templateKey,
          channel,
          success,
          recipient: payload.clientName
        }
      });

      return success;
    } catch (error) {
      console.error('Failed to send reminder:', error);
      return false;
    }
  }

  private static renderTemplate(template: ReminderTemplate, payload: ReminderPayload): {
    subject: string;
    body: string;
  } {
    const replacePlaceholders = (text: string): string => {
      return text
        .replace(/{{clientName}}/g, payload.clientName)
        .replace(/{{ay}}/g, payload.ay)
        .replace(/{{dueDate}}/g, payload.dueDate || '')
        .replace(/{{firmName}}/g, payload.firmName)
        .replace(/{{link}}/g, payload.link || '')
        .replace(/{{hoursRemaining}}/g, payload.hoursRemaining?.toString() || '')
        .replace(/{{filedAt}}/g, payload.filedAt || '');
    };

    return {
      subject: replacePlaceholders(template.subject),
      body: replacePlaceholders(template.body)
    };
  }

  private static async sendEmail(recipient: string, subject: string, body: string): Promise<boolean> {
    // Email sending logic using SMTP
    const smtpConfig = {
      host: process.env.SMTP_HOST,
      port: parseInt(process.env.SMTP_PORT || '587'),
      user: process.env.SMTP_USER,
      pass: process.env.SMTP_PASS
    };

    if (!smtpConfig.host) {
      console.log('SMTP not configured, email would be sent:', { recipient, subject, body });
      return true; // Return true for demo purposes
    }

    // TODO: Implement actual SMTP sending
    console.log('Email sent:', { recipient, subject, body });
    return true;
  }

  private static async sendWhatsApp(recipient: string, message: string): Promise<boolean> {
    // WhatsApp sending using WATI/Twilio API
    const whatsappConfig = {
      apiKey: process.env.WHATSAPP_API_KEY,
      endpoint: process.env.WHATSAPP_ENDPOINT
    };

    if (!whatsappConfig.apiKey) {
      console.log('WhatsApp not configured, message would be sent:', { recipient, message });
      return true; // Return true for demo purposes
    }

    // TODO: Implement actual WhatsApp API call
    console.log('WhatsApp sent:', { recipient, message });
    return true;
  }

  private static async sendSMS(recipient: string, message: string): Promise<boolean> {
    // SMS sending using Twilio or similar
    const smsConfig = {
      apiKey: process.env.SMS_API_KEY,
      endpoint: process.env.SMS_ENDPOINT
    };

    if (!smsConfig.apiKey) {
      console.log('SMS not configured, message would be sent:', { recipient, message });
      return true; // Return true for demo purposes
    }

    // TODO: Implement actual SMS API call
    console.log('SMS sent:', { recipient, message });
    return true;
  }

  static getTemplates(): ReminderTemplate[] {
    return this.TEMPLATES;
  }

  static getTemplate(key: string): ReminderTemplate | undefined {
    return this.TEMPLATES.find(t => t.key === key);
  }

  static async checkSLABreaches(): Promise<void> {
    // This would be called by a scheduled job to check for SLA breaches
    // and send automatic reminders
    
    // TODO: Implement SLA breach checking logic
    console.log('Checking SLA breaches...');
  }
}
