import { storage } from '../storage';
import { ActivityLogService } from './activityLog';
import type { Engagement } from '@shared/schema';

export type EngagementStatus = 
  | "DATA_REQUESTED"
  | "DATA_RECEIVED_PRELIMINARY" 
  | "KYC_CONSENT_CAPTURED"
  | "DOC_INDEXING_VETTING"
  | "AIS_TIS_IMPORTED_RECONCILED"
  | "COMPUTATION_PREPARED"
  | "MANAGER_REVIEW"
  | "PARTNER_REVIEW" 
  | "CLIENT_REVIEW_SIGNOFF"
  | "ITR_CREATED_JSON_VALIDATED"
  | "ITR_FILED_ACK_CAPTURED"
  | "E_VERIFICATION_PENDING"
  | "E_VERIFICATION_DONE"
  | "CPC_PROCESSING"
  | "REFUND_ADJUSTMENT_UPDATE"
  | "CLOSED";

export interface StatusTransition {
  from: EngagementStatus;
  to: EngagementStatus;
  requiredRole?: string[];
  requiredPermissions?: string[];
  validationRules?: ((engagement: Engagement) => boolean)[];
}

export class WorkflowEngine {
  private static readonly WORKFLOW_TRANSITIONS: StatusTransition[] = [
    {
      from: "DATA_REQUESTED",
      to: "DATA_RECEIVED_PRELIMINARY",
      requiredRole: ["STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "DATA_RECEIVED_PRELIMINARY",
      to: "KYC_CONSENT_CAPTURED",
      requiredRole: ["STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "KYC_CONSENT_CAPTURED",
      to: "DOC_INDEXING_VETTING",
      requiredRole: ["STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "DOC_INDEXING_VETTING",
      to: "AIS_TIS_IMPORTED_RECONCILED",
      requiredRole: ["STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "AIS_TIS_IMPORTED_RECONCILED",
      to: "COMPUTATION_PREPARED",
      requiredRole: ["STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "COMPUTATION_PREPARED",
      to: "MANAGER_REVIEW",
      requiredRole: ["STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "MANAGER_REVIEW",
      to: "PARTNER_REVIEW",
      requiredRole: ["MANAGER", "PARTNER"]
    },
    {
      from: "MANAGER_REVIEW",
      to: "COMPUTATION_PREPARED", // Back to computation if changes requested
      requiredRole: ["MANAGER", "PARTNER"]
    },
    {
      from: "PARTNER_REVIEW",
      to: "CLIENT_REVIEW_SIGNOFF",
      requiredRole: ["PARTNER"]
    },
    {
      from: "PARTNER_REVIEW",
      to: "COMPUTATION_PREPARED", // Back to computation if changes requested
      requiredRole: ["PARTNER"]
    },
    {
      from: "CLIENT_REVIEW_SIGNOFF",
      to: "ITR_CREATED_JSON_VALIDATED",
      requiredRole: ["CLIENT", "STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "ITR_CREATED_JSON_VALIDATED",
      to: "ITR_FILED_ACK_CAPTURED",
      requiredRole: ["STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "ITR_FILED_ACK_CAPTURED",
      to: "E_VERIFICATION_PENDING",
      requiredRole: ["CLIENT", "STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "E_VERIFICATION_PENDING",
      to: "E_VERIFICATION_DONE",
      requiredRole: ["CLIENT", "STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "E_VERIFICATION_DONE",
      to: "CPC_PROCESSING",
      requiredRole: ["STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "CPC_PROCESSING",
      to: "REFUND_ADJUSTMENT_UPDATE",
      requiredRole: ["STAFF", "MANAGER", "PARTNER"]
    },
    {
      from: "REFUND_ADJUSTMENT_UPDATE",
      to: "CLOSED",
      requiredRole: ["MANAGER", "PARTNER"]
    }
  ];

  static async transitionStatus(
    engagementId: string,
    newStatus: EngagementStatus,
    userId: string,
    userRole: string,
    reason?: string,
    ip?: string
  ): Promise<Engagement> {
    const engagement = await storage.getEngagement(engagementId);
    
    if (!engagement) {
      throw new Error('Engagement not found');
    }

    const currentStatus = engagement.status as EngagementStatus;
    
    // Find valid transition
    const transition = this.WORKFLOW_TRANSITIONS.find(t => 
      t.from === currentStatus && t.to === newStatus
    );

    if (!transition) {
      throw new Error(`Invalid status transition from ${currentStatus} to ${newStatus}`);
    }

    // Check role permissions
    if (transition.requiredRole && !transition.requiredRole.includes(userRole)) {
      throw new Error(`Role ${userRole} is not authorized for this transition`);
    }

    // Run validation rules
    if (transition.validationRules) {
      for (const rule of transition.validationRules) {
        if (!rule(engagement)) {
          throw new Error('Validation failed for status transition');
        }
      }
    }

    // Log the transition
    await ActivityLogService.logActivity({
      actorUserId: userId,
      actorRole: userRole,
      action: ActivityLogService.ACTIONS.STATUS_CHANGE,
      entityType: ActivityLogService.ENTITY_TYPES.ENGAGEMENT,
      entityId: engagementId,
      ip: ip || 'unknown',
      beforeData: { status: currentStatus },
      afterData: { status: newStatus, reason }
    });

    // Update engagement status
    const updatedEngagement = await storage.updateEngagement(engagementId, {
      status: newStatus
    });

    // Check for SLA implications
    await this.checkSLA(engagementId, newStatus);

    return updatedEngagement;
  }

  static getValidTransitions(currentStatus: EngagementStatus, userRole: string): EngagementStatus[] {
    return this.WORKFLOW_TRANSITIONS
      .filter(t => 
        t.from === currentStatus && 
        (!t.requiredRole || t.requiredRole.includes(userRole))
      )
      .map(t => t.to);
  }

  static getStatusDisplayName(status: EngagementStatus): string {
    const displayNames: Record<EngagementStatus, string> = {
      "DATA_REQUESTED": "Data Requested",
      "DATA_RECEIVED_PRELIMINARY": "Data Received",
      "KYC_CONSENT_CAPTURED": "KYC/Consent Captured",
      "DOC_INDEXING_VETTING": "Document Vetting",
      "AIS_TIS_IMPORTED_RECONCILED": "AIS/TIS Reconciled",
      "COMPUTATION_PREPARED": "Computation Prepared",
      "MANAGER_REVIEW": "Manager Review",
      "PARTNER_REVIEW": "Partner Review",
      "CLIENT_REVIEW_SIGNOFF": "Client Sign-off",
      "ITR_CREATED_JSON_VALIDATED": "ITR Created",
      "ITR_FILED_ACK_CAPTURED": "ITR Filed",
      "E_VERIFICATION_PENDING": "E-Verification Pending",
      "E_VERIFICATION_DONE": "E-Verification Done",
      "CPC_PROCESSING": "CPC Processing",
      "REFUND_ADJUSTMENT_UPDATE": "Refund Update",
      "CLOSED": "Closed"
    };

    return displayNames[status] || status;
  }

  static getStatusColor(status: EngagementStatus): string {
    const colors: Record<EngagementStatus, string> = {
      "DATA_REQUESTED": "gray",
      "DATA_RECEIVED_PRELIMINARY": "blue",
      "KYC_CONSENT_CAPTURED": "indigo",
      "DOC_INDEXING_VETTING": "purple",
      "AIS_TIS_IMPORTED_RECONCILED": "purple",
      "COMPUTATION_PREPARED": "yellow",
      "MANAGER_REVIEW": "orange",
      "PARTNER_REVIEW": "red",
      "CLIENT_REVIEW_SIGNOFF": "pink",
      "ITR_CREATED_JSON_VALIDATED": "emerald",
      "ITR_FILED_ACK_CAPTURED": "green",
      "E_VERIFICATION_PENDING": "yellow",
      "E_VERIFICATION_DONE": "green",
      "CPC_PROCESSING": "blue",
      "REFUND_ADJUSTMENT_UPDATE": "green",
      "CLOSED": "emerald"
    };

    return colors[status] || "gray";
  }

  private static async checkSLA(engagementId: string, newStatus: EngagementStatus): Promise<void> {
    // TODO: Implement SLA checking logic
    // This would calculate if the transition breaches any SLA targets
    // and create SLA events accordingly
  }

  static async autoCreateChecklist(engagementId: string, itrFormType: string): Promise<void> {
    const engagement = await storage.getEngagement(engagementId);
    if (!engagement) return;

    // Create checklist based on ITR form type
    const checklistItems = this.getChecklistTemplate(itrFormType);

    // TODO: Create checklist and items in database
    // This would involve creating a checklist record and associated items
  }

  private static getChecklistTemplate(itrFormType: string): Array<{
    item: string;
    sectionRef?: string;
    mandatory: boolean;
  }> {
    const templates: Record<string, Array<{
      item: string;
      sectionRef?: string;
      mandatory: boolean;
    }>> = {
      "ITR-1": [
        { item: "PAN-Aadhaar link verification", mandatory: true },
        { item: "26AS downloaded and reviewed", sectionRef: "Sec 192", mandatory: true },
        { item: "AIS/TIS data imported & reconciled", sectionRef: "Sec 285BA", mandatory: true },
        { item: "Form 16 obtained and verified", mandatory: true },
        { item: "Interest income from banks verified", mandatory: false },
        { item: "80C deductions verified with proofs", sectionRef: "Sch VI-A", mandatory: false },
        { item: "80D health insurance verified", sectionRef: "Sch VI-A", mandatory: false }
      ],
      "ITR-2": [
        { item: "PAN-Aadhaar link verification", mandatory: true },
        { item: "26AS downloaded and reviewed", sectionRef: "Sec 192", mandatory: true },
        { item: "AIS/TIS data imported & reconciled", sectionRef: "Sec 285BA", mandatory: true },
        { item: "Form 16 obtained and verified", mandatory: true },
        { item: "Capital gains computation verified", sectionRef: "Sch CG", mandatory: false },
        { item: "House property schedule completed", sectionRef: "Sch HP", mandatory: false },
        { item: "80C deductions verified with proofs", sectionRef: "Sch VI-A", mandatory: false },
        { item: "Foreign income/assets disclosed", sectionRef: "Sch FA", mandatory: false }
      ]
      // Add more templates for ITR-3, ITR-4, etc.
    };

    return templates[itrFormType] || [];
  }
}
