import { neon } from "@neondatabase/serverless";
import { drizzle } from "drizzle-orm/neon-http";
import { eq, and, or, desc, asc, sql, count, like, inArray } from "drizzle-orm";
import * as schema from "@shared/schema";
import type {
  User,
  InsertUser,
  Client,
  InsertClient,
  Engagement,
  InsertEngagement,
  Document,
  InsertDocument,
  Comment,
  InsertComment,
  ChecklistItem,
  InsertChecklistItem,
  Computation,
  InsertComputation,
  ActivityLogEntry
} from "@shared/schema";

// Database connection
const connectionString = process.env.DATABASE_URL;
if (!connectionString) {
  throw new Error("DATABASE_URL environment variable is required");
}

const client = neon(connectionString);
const db = drizzle(client, { schema });

export interface IStorage {
  // Users
  getUser(id: string): Promise<User | undefined>;
  getUserByEmail(email: string): Promise<User | undefined>;
  createUser(user: InsertUser): Promise<User>;
  updateUser(id: string, updates: Partial<User>): Promise<User>;
  getUsers(): Promise<User[]>;

  // Clients
  getClient(id: string): Promise<Client | undefined>;
  getClients(filters?: { search?: string; limit?: number; offset?: number }): Promise<{ clients: Client[]; total: number }>;
  createClient(client: InsertClient): Promise<Client>;
  updateClient(id: string, updates: Partial<Client>): Promise<Client>;
  deleteClient(id: string): Promise<void>;

  // Engagements
  getEngagement(id: string): Promise<Engagement | undefined>;
  getEngagements(filters?: { 
    clientId?: string; 
    status?: string; 
    assigneeId?: string; 
    search?: string; 
    limit?: number; 
    offset?: number;
  }): Promise<{ engagements: (Engagement & { client: Client })[], total: number }>;
  createEngagement(engagement: InsertEngagement): Promise<Engagement>;
  updateEngagement(id: string, updates: Partial<Engagement>): Promise<Engagement>;
  deleteEngagement(id: string): Promise<void>;

  // Documents
  getDocument(id: string): Promise<Document | undefined>;
  getDocumentsByEngagement(engagementId: string): Promise<Document[]>;
  createDocument(document: InsertDocument): Promise<Document>;
  deleteDocument(id: string): Promise<void>;

  // Comments
  getCommentsByEngagement(engagementId: string): Promise<(Comment & { author: User })[]>;
  createComment(comment: InsertComment): Promise<Comment>;

  // Checklist Items
  getChecklistItemsByEngagement(engagementId: string): Promise<ChecklistItem[]>;
  createChecklistItem(item: InsertChecklistItem): Promise<ChecklistItem>;
  updateChecklistItem(id: string, updates: Partial<ChecklistItem>): Promise<ChecklistItem>;

  // Computations
  getComputationsByEngagement(engagementId: string): Promise<Computation[]>;
  createComputation(computation: InsertComputation): Promise<Computation>;
  updateComputation(id: string, updates: Partial<Computation>): Promise<Computation>;

  // Activity Log
  createActivityLog(entry: Omit<ActivityLogEntry, 'id' | 'createdAt'>): Promise<ActivityLogEntry>;
  getActivityLog(filters?: { 
    engagementId?: string; 
    entityType?: string; 
    limit?: number; 
    offset?: number;
  }): Promise<{ logs: ActivityLogEntry[], total: number }>;

  // Dashboard/Analytics
  getDashboardStats(userId?: string): Promise<{
    totalEngagements: number;
    pendingReviews: number;
    filedReturns: number;
    slaBreaches: number;
  }>;
  
  getPipelineStats(): Promise<Array<{ status: string; count: number }>>;
  getRecentActivity(limit?: number): Promise<Array<ActivityLogEntry & { actor: User | null }>>;
}

export class DatabaseStorage implements IStorage {
  async getUser(id: string): Promise<User | undefined> {
    const result = await db.select().from(schema.users).where(eq(schema.users.id, id)).limit(1);
    return result[0];
  }

  async getUserByEmail(email: string): Promise<User | undefined> {
    const result = await db.select().from(schema.users).where(eq(schema.users.email, email)).limit(1);
    return result[0];
  }

  async createUser(user: InsertUser): Promise<User> {
    const result = await db.insert(schema.users).values(user).returning();
    return result[0];
  }

  async updateUser(id: string, updates: Partial<User>): Promise<User> {
    const result = await db.update(schema.users)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(schema.users.id, id))
      .returning();
    return result[0];
  }

  async getUsers(): Promise<User[]> {
    return await db.select().from(schema.users).orderBy(asc(schema.users.name));
  }

  async getClient(id: string): Promise<Client | undefined> {
    const result = await db.select().from(schema.clients).where(eq(schema.clients.id, id)).limit(1);
    return result[0];
  }

  async getClients(filters?: { search?: string; limit?: number; offset?: number }): Promise<{ clients: Client[]; total: number }> {
    const limit = filters?.limit || 50;
    const offset = filters?.offset || 0;
    
    let query = db.select().from(schema.clients);
    let countQuery = db.select({ count: count() }).from(schema.clients);

    if (filters?.search) {
      const searchCondition = or(
        like(schema.clients.displayName, `%${filters.search}%`),
        like(schema.clients.pan, `%${filters.search}%`)
      );
      query = query.where(searchCondition);
      countQuery = countQuery.where(searchCondition);
    }

    const [clients, totalResult] = await Promise.all([
      query.orderBy(asc(schema.clients.displayName)).limit(limit).offset(offset),
      countQuery
    ]);

    return {
      clients,
      total: totalResult[0].count
    };
  }

  async createClient(client: InsertClient): Promise<Client> {
    const result = await db.insert(schema.clients).values(client).returning();
    return result[0];
  }

  async updateClient(id: string, updates: Partial<Client>): Promise<Client> {
    const result = await db.update(schema.clients)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(schema.clients.id, id))
      .returning();
    return result[0];
  }

  async deleteClient(id: string): Promise<void> {
    await db.delete(schema.clients).where(eq(schema.clients.id, id));
  }

  async getEngagement(id: string): Promise<Engagement | undefined> {
    const result = await db.select().from(schema.engagements).where(eq(schema.engagements.id, id)).limit(1);
    return result[0];
  }

  async getEngagements(filters?: { 
    clientId?: string; 
    status?: string; 
    assigneeId?: string; 
    search?: string; 
    limit?: number; 
    offset?: number;
  }): Promise<{ engagements: (Engagement & { client: Client })[], total: number }> {
    const limit = filters?.limit || 50;
    const offset = filters?.offset || 0;

    let query = db.select({
      ...schema.engagements,
      client: schema.clients
    }).from(schema.engagements)
      .innerJoin(schema.clients, eq(schema.engagements.clientId, schema.clients.id));

    let countQuery = db.select({ count: count() }).from(schema.engagements)
      .innerJoin(schema.clients, eq(schema.engagements.clientId, schema.clients.id));

    const conditions = [];

    if (filters?.clientId) {
      conditions.push(eq(schema.engagements.clientId, filters.clientId));
    }

    if (filters?.status) {
      conditions.push(eq(schema.engagements.status, filters.status as any));
    }

    if (filters?.search) {
      conditions.push(or(
        like(schema.clients.displayName, `%${filters.search}%`),
        like(schema.clients.pan, `%${filters.search}%`),
        like(schema.engagements.ay, `%${filters.search}%`)
      ));
    }

    if (conditions.length > 0) {
      const whereCondition = conditions.length === 1 ? conditions[0] : and(...conditions);
      query = query.where(whereCondition);
      countQuery = countQuery.where(whereCondition);
    }

    const [engagements, totalResult] = await Promise.all([
      query.orderBy(desc(schema.engagements.updatedAt)).limit(limit).offset(offset),
      countQuery
    ]);

    return {
      engagements,
      total: totalResult[0].count
    };
  }

  async createEngagement(engagement: InsertEngagement): Promise<Engagement> {
    const result = await db.insert(schema.engagements).values(engagement).returning();
    return result[0];
  }

  async updateEngagement(id: string, updates: Partial<Engagement>): Promise<Engagement> {
    const result = await db.update(schema.engagements)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(schema.engagements.id, id))
      .returning();
    return result[0];
  }

  async deleteEngagement(id: string): Promise<void> {
    await db.delete(schema.engagements).where(eq(schema.engagements.id, id));
  }

  async getDocument(id: string): Promise<Document | undefined> {
    const result = await db.select().from(schema.documents).where(eq(schema.documents.id, id)).limit(1);
    return result[0];
  }

  async getDocumentsByEngagement(engagementId: string): Promise<Document[]> {
    return await db.select().from(schema.documents)
      .where(eq(schema.documents.engagementId, engagementId))
      .orderBy(desc(schema.documents.createdAt));
  }

  async createDocument(document: InsertDocument): Promise<Document> {
    const result = await db.insert(schema.documents).values(document).returning();
    return result[0];
  }

  async deleteDocument(id: string): Promise<void> {
    await db.delete(schema.documents).where(eq(schema.documents.id, id));
  }

  async getCommentsByEngagement(engagementId: string): Promise<(Comment & { author: User })[]> {
    return await db.select({
      ...schema.comments,
      author: schema.users
    }).from(schema.comments)
      .innerJoin(schema.users, eq(schema.comments.authorId, schema.users.id))
      .where(eq(schema.comments.engagementId, engagementId))
      .orderBy(asc(schema.comments.createdAt));
  }

  async createComment(comment: InsertComment): Promise<Comment> {
    const result = await db.insert(schema.comments).values(comment).returning();
    return result[0];
  }

  async getChecklistItemsByEngagement(engagementId: string): Promise<ChecklistItem[]> {
    return await db.select().from(schema.checklistItems)
      .innerJoin(schema.checklists, eq(schema.checklistItems.checklistId, schema.checklists.id))
      .where(eq(schema.checklists.engagementId, engagementId))
      .orderBy(asc(schema.checklistItems.createdAt));
  }

  async createChecklistItem(item: InsertChecklistItem): Promise<ChecklistItem> {
    const result = await db.insert(schema.checklistItems).values(item).returning();
    return result[0];
  }

  async updateChecklistItem(id: string, updates: Partial<ChecklistItem>): Promise<ChecklistItem> {
    const result = await db.update(schema.checklistItems)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(schema.checklistItems.id, id))
      .returning();
    return result[0];
  }

  async getComputationsByEngagement(engagementId: string): Promise<Computation[]> {
    return await db.select().from(schema.computations)
      .where(eq(schema.computations.engagementId, engagementId))
      .orderBy(desc(schema.computations.createdAt));
  }

  async createComputation(computation: InsertComputation): Promise<Computation> {
    const result = await db.insert(schema.computations).values(computation).returning();
    return result[0];
  }

  async updateComputation(id: string, updates: Partial<Computation>): Promise<Computation> {
    const result = await db.update(schema.computations)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(schema.computations.id, id))
      .returning();
    return result[0];
  }

  async createActivityLog(entry: Omit<ActivityLogEntry, 'id' | 'createdAt'>): Promise<ActivityLogEntry> {
    const result = await db.insert(schema.activityLog).values(entry).returning();
    return result[0];
  }

  async getActivityLog(filters?: { 
    engagementId?: string; 
    entityType?: string; 
    limit?: number; 
    offset?: number;
  }): Promise<{ logs: ActivityLogEntry[], total: number }> {
    const limit = filters?.limit || 50;
    const offset = filters?.offset || 0;

    let query = db.select().from(schema.activityLog);
    let countQuery = db.select({ count: count() }).from(schema.activityLog);

    const conditions = [];

    if (filters?.engagementId) {
      conditions.push(eq(schema.activityLog.entityId, filters.engagementId));
    }

    if (filters?.entityType) {
      conditions.push(eq(schema.activityLog.entityType, filters.entityType));
    }

    if (conditions.length > 0) {
      const whereCondition = conditions.length === 1 ? conditions[0] : and(...conditions);
      query = query.where(whereCondition);
      countQuery = countQuery.where(whereCondition);
    }

    const [logs, totalResult] = await Promise.all([
      query.orderBy(desc(schema.activityLog.createdAt)).limit(limit).offset(offset),
      countQuery
    ]);

    return {
      logs,
      total: totalResult[0].count
    };
  }

  async getDashboardStats(userId?: string): Promise<{
    totalEngagements: number;
    pendingReviews: number;
    filedReturns: number;
    slaBreaches: number;
  }> {
    const [
      totalEngagementsResult,
      pendingReviewsResult,
      filedReturnsResult,
      slaBreachesResult
    ] = await Promise.all([
      db.select({ count: count() }).from(schema.engagements),
      db.select({ count: count() }).from(schema.engagements)
        .where(or(
          eq(schema.engagements.status, "MANAGER_REVIEW"),
          eq(schema.engagements.status, "PARTNER_REVIEW")
        )),
      db.select({ count: count() }).from(schema.engagements)
        .where(inArray(schema.engagements.status, [
          "ITR_FILED_ACK_CAPTURED",
          "E_VERIFICATION_DONE",
          "CPC_PROCESSING",
          "REFUND_ADJUSTMENT_UPDATE",
          "CLOSED"
        ])),
      db.select({ count: count() }).from(schema.slaEvents)
        .where(eq(schema.slaEvents.breached, true))
    ]);

    return {
      totalEngagements: totalEngagementsResult[0].count,
      pendingReviews: pendingReviewsResult[0].count,
      filedReturns: filedReturnsResult[0].count,
      slaBreaches: slaBreachesResult[0].count
    };
  }

  async getPipelineStats(): Promise<Array<{ status: string; count: number }>> {
    const result = await db.select({
      status: schema.engagements.status,
      count: count()
    }).from(schema.engagements)
      .groupBy(schema.engagements.status);

    return result.map(r => ({ status: r.status, count: r.count }));
  }

  async getRecentActivity(limit = 10): Promise<Array<ActivityLogEntry & { actor: User | null }>> {
    return await db.select({
      ...schema.activityLog,
      actor: schema.users
    }).from(schema.activityLog)
      .leftJoin(schema.users, eq(schema.activityLog.actorUserId, schema.users.id))
      .orderBy(desc(schema.activityLog.createdAt))
      .limit(limit);
  }
}

export const storage = new DatabaseStorage();
