import { randomBytes, scrypt, timingSafeEqual } from 'crypto';
import { promisify } from 'util';

const scryptAsync = promisify(scrypt);

export class CryptoUtils {
  /**
   * Hash a password using scrypt
   */
  static async hashPassword(password: string): Promise<string> {
    const salt = randomBytes(32);
    const derivedKey = await scryptAsync(password, salt, 64) as Buffer;
    return salt.toString('hex') + ':' + derivedKey.toString('hex');
  }

  /**
   * Verify a password against a hash
   */
  static async verifyPassword(password: string, hash: string): Promise<boolean> {
    const [saltHex, keyHex] = hash.split(':');
    const salt = Buffer.from(saltHex, 'hex');
    const key = Buffer.from(keyHex, 'hex');
    const derivedKey = await scryptAsync(password, salt, 64) as Buffer;
    
    return timingSafeEqual(key, derivedKey);
  }

  /**
   * Generate a secure random token
   */
  static generateToken(length: number = 32): string {
    return randomBytes(length).toString('hex');
  }

  /**
   * Mask PAN number for display (e.g., ABCDE1234F -> AB***1234F)
   */
  static maskPAN(pan: string): string {
    if (!pan || pan.length !== 10) {
      return pan;
    }
    return pan.substring(0, 2) + '***' + pan.substring(5);
  }

  /**
   * Mask Aadhaar number for display (e.g., 123456789012 -> ****9012)
   */
  static maskAadhaar(aadhaar: string): string {
    if (!aadhaar || aadhaar.length !== 12) {
      return aadhaar;
    }
    return '****' + aadhaar.substring(8);
  }

  /**
   * Validate PAN format
   */
  static validatePAN(pan: string): boolean {
    const panRegex = /^[A-Z]{5}[0-9]{4}[A-Z]{1}$/;
    return panRegex.test(pan);
  }

  /**
   * Validate Aadhaar format
   */
  static validateAadhaar(aadhaar: string): boolean {
    const aadhaarRegex = /^[0-9]{12}$/;
    return aadhaarRegex.test(aadhaar);
  }

  /**
   * Generate a unique engagement number
   */
  static generateEngagementNumber(clientCode: string, ay: string): string {
    const year = ay.split('-')[0];
    const timestamp = Date.now().toString(36).toUpperCase();
    return `ENG-${clientCode}-${year}-${timestamp}`;
  }

  /**
   * Sanitize filename for safe storage
   */
  static sanitizeFilename(filename: string): string {
    return filename
      .replace(/[^a-zA-Z0-9.-]/g, '_')
      .replace(/_{2,}/g, '_')
      .replace(/^_|_$/g, '');
  }
}
