import { z } from 'zod';
import { CryptoUtils } from './crypto';

// Common validation patterns
export const panSchema = z.string()
  .length(10, 'PAN must be 10 characters')
  .regex(/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/, 'Invalid PAN format')
  .refine(CryptoUtils.validatePAN, 'Invalid PAN format');

export const aadhaarSchema = z.string()
  .length(12, 'Aadhaar must be 12 digits')
  .regex(/^[0-9]{12}$/, 'Aadhaar must contain only digits')
  .refine(CryptoUtils.validateAadhaar, 'Invalid Aadhaar format');

export const emailSchema = z.string()
  .email('Invalid email format')
  .max(255, 'Email too long');

export const mobileSchema = z.string()
  .regex(/^[6-9]\d{9}$/, 'Invalid Indian mobile number format')
  .optional();

export const aySchema = z.string()
  .regex(/^\d{4}-\d{2}$/, 'Assessment Year must be in format YYYY-YY (e.g., 2024-25)');

// File validation
export const fileUploadSchema = z.object({
  filename: z.string().min(1, 'Filename is required'),
  mimetype: z.enum([
    'application/pdf',
    'application/vnd.ms-excel',
    'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'text/csv',
    'image/jpeg',
    'image/png',
    'image/jpg'
  ], { message: 'Invalid file type' }),
  size: z.number().max(10 * 1024 * 1024, 'File size must be less than 10MB')
});

// Client validation
export const createClientSchema = z.object({
  displayName: z.string().min(1, 'Display name is required').max(255),
  legalName: z.string().min(1, 'Legal name is required').max(255),
  pan: panSchema,
  aadhaarLast4: z.string().length(4, 'Last 4 digits of Aadhaar required').regex(/^\d{4}$/).optional(),
  dateOfBirth: z.string().datetime().optional(),
  incorporationDate: z.string().datetime().optional(),
  address: z.object({
    line1: z.string().optional(),
    line2: z.string().optional(),
    city: z.string().optional(),
    state: z.string().optional(),
    pincode: z.string().regex(/^\d{6}$/).optional(),
    country: z.string().default('India')
  }).optional(),
  contacts: z.array(z.object({
    type: z.enum(['primary', 'alternate', 'business']),
    email: emailSchema.optional(),
    mobile: mobileSchema
  })).default([]),
  riskCategory: z.enum(['LOW', 'MEDIUM', 'HIGH']).default('MEDIUM'),
  tags: z.array(z.string()).default([])
});

// Engagement validation
export const createEngagementSchema = z.object({
  clientId: z.string().uuid('Invalid client ID'),
  ay: aySchema,
  itrFormType: z.enum(['ITR-1', 'ITR-2', 'ITR-3', 'ITR-4', 'ITR-5', 'ITR-6', 'ITR-7']),
  tier: z.enum(['INDIVIDUAL', 'HUF', 'FIRM', 'COMPANY', 'TRUST']),
  isAuditApplicable: z.boolean().default(false),
  priority: z.enum(['LOW', 'MEDIUM', 'HIGH', 'CRITICAL']).default('MEDIUM'),
  dueDates: z.object({
    filingDue: z.string().datetime().optional(),
    auditDue: z.string().datetime().optional()
  }).default({})
});

// Comment validation
export const createCommentSchema = z.object({
  engagementId: z.string().uuid('Invalid engagement ID'),
  entityType: z.string().min(1, 'Entity type is required'),
  entityId: z.string().uuid('Invalid entity ID'),
  body: z.string().min(1, 'Comment body is required').max(2000, 'Comment too long'),
  mentions: z.array(z.string().uuid()).default([])
});

// Checklist item validation
export const updateChecklistItemSchema = z.object({
  status: z.enum(['PENDING', 'IN_PROGRESS', 'DONE', 'NA']),
  notes: z.string().max(1000, 'Notes too long').optional(),
  assigneeId: z.string().uuid('Invalid assignee ID').optional()
});

// User registration validation
export const registerUserSchema = z.object({
  name: z.string().min(1, 'Name is required').max(255),
  email: emailSchema,
  mobile: mobileSchema,
  password: z.string()
    .min(8, 'Password must be at least 8 characters')
    .regex(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/, 'Password must contain at least one lowercase letter, one uppercase letter, and one number'),
  role: z.enum(['PARTNER', 'MANAGER', 'STAFF', 'CLIENT', 'AUDITOR'])
});

// Login validation
export const loginSchema = z.object({
  email: emailSchema,
  password: z.string().min(1, 'Password is required')
});

// Status transition validation
export const statusTransitionSchema = z.object({
  status: z.enum([
    'DATA_REQUESTED',
    'DATA_RECEIVED_PRELIMINARY',
    'KYC_CONSENT_CAPTURED',
    'DOC_INDEXING_VETTING',
    'AIS_TIS_IMPORTED_RECONCILED',
    'COMPUTATION_PREPARED',
    'MANAGER_REVIEW',
    'PARTNER_REVIEW',
    'CLIENT_REVIEW_SIGNOFF',
    'ITR_CREATED_JSON_VALIDATED',
    'ITR_FILED_ACK_CAPTURED',
    'E_VERIFICATION_PENDING',
    'E_VERIFICATION_DONE',
    'CPC_PROCESSING',
    'REFUND_ADJUSTMENT_UPDATE',
    'CLOSED'
  ]),
  reason: z.string().max(500, 'Reason too long').optional()
});

// Computation validation
export const createComputationSchema = z.object({
  engagementId: z.string().uuid('Invalid engagement ID'),
  engine: z.enum(['excel', 'python']),
  version: z.string().default('1.0'),
  summaryJson: z.object({
    totalIncome: z.number().optional(),
    totalDeductions: z.number().optional(),
    taxableIncome: z.number().optional(),
    totalTax: z.number().optional(),
    surcharge: z.number().optional(),
    cess: z.number().optional(),
    totalTaxPayable: z.number().optional()
  }),
  attachments: z.array(z.string()).default([]),
  remarks: z.string().max(2000, 'Remarks too long').optional()
});

// Search/filter validation
export const searchSchema = z.object({
  q: z.string().optional(),
  limit: z.number().min(1).max(100).default(50),
  offset: z.number().min(0).default(0),
  status: z.string().optional(),
  priority: z.string().optional(),
  assignee: z.string().uuid().optional(),
  ay: z.string().optional()
});

// Validation middleware helper
export function validateSchema<T>(schema: z.ZodSchema<T>) {
  return (req: any, res: any, next: any) => {
    try {
      const validatedData = schema.parse(req.body);
      req.validatedData = validatedData;
      next();
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({
          message: 'Validation failed',
          errors: error.errors.map(err => ({
            field: err.path.join('.'),
            message: err.message
          }))
        });
      }
      next(error);
    }
  };
}

export function validateQuery<T>(schema: z.ZodSchema<T>) {
  return (req: any, res: any, next: any) => {
    try {
      const validatedQuery = schema.parse(req.query);
      req.validatedQuery = validatedQuery;
      next();
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({
          message: 'Query validation failed',
          errors: error.errors.map(err => ({
            field: err.path.join('.'),
            message: err.message
          }))
        });
      }
      next(error);
    }
  };
}
