import { sql } from "drizzle-orm";
import { pgTable, text, varchar, timestamp, boolean, integer, jsonb, uuid, decimal, pgEnum } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

// Enums
export const userRoleEnum = pgEnum("user_role", ["PARTNER", "MANAGER", "STAFF", "CLIENT", "AUDITOR"]);
export const engagementStatusEnum = pgEnum("engagement_status", [
  "DATA_REQUESTED",
  "DATA_RECEIVED_PRELIMINARY", 
  "KYC_CONSENT_CAPTURED",
  "DOC_INDEXING_VETTING",
  "AIS_TIS_IMPORTED_RECONCILED",
  "COMPUTATION_PREPARED",
  "MANAGER_REVIEW",
  "PARTNER_REVIEW", 
  "CLIENT_REVIEW_SIGNOFF",
  "ITR_CREATED_JSON_VALIDATED",
  "ITR_FILED_ACK_CAPTURED",
  "E_VERIFICATION_PENDING",
  "E_VERIFICATION_DONE",
  "CPC_PROCESSING",
  "REFUND_ADJUSTMENT_UPDATE",
  "CLOSED"
]);
export const priorityEnum = pgEnum("priority", ["LOW", "MEDIUM", "HIGH", "CRITICAL"]);
export const itrFormTypeEnum = pgEnum("itr_form_type", ["ITR-1", "ITR-2", "ITR-3", "ITR-4", "ITR-5", "ITR-6", "ITR-7"]);
export const tierEnum = pgEnum("tier", ["INDIVIDUAL", "HUF", "FIRM", "COMPANY", "TRUST"]);
export const checklistStatusEnum = pgEnum("checklist_status", ["PENDING", "IN_PROGRESS", "DONE", "NA"]);
export const reviewStatusEnum = pgEnum("review_status", ["PENDING", "APPROVED", "REJECTED", "CHANGES_REQUESTED"]);
export const eVerifyStatusEnum = pgEnum("e_verify_status", ["PENDING", "DONE", "BY_AADHAAR_OTP", "BY_BANK", "BY_DSC"]);
export const cpcStatusEnum = pgEnum("cpc_status", ["PROCESSED", "DEFECTIVE", "NOTICE_ISSUED"]);
export const refundStatusEnum = pgEnum("refund_status", ["ISSUED", "PENDING", "ADJUSTED"]);
export const reminderChannelEnum = pgEnum("reminder_channel", ["EMAIL", "WHATSAPP", "SMS"]);
export const reminderStatusEnum = pgEnum("reminder_status", ["SCHEDULED", "SENT", "FAILED"]);

// Core tables
export const users = pgTable("users", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  email: text("email").notNull().unique(),
  mobile: text("mobile"),
  password: text("password").notNull(),
  role: userRoleEnum("role").notNull(),
  isActive: boolean("is_active").default(true),
  lastLoginAt: timestamp("last_login_at"),
  twoFASecret: text("two_fa_secret"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

export const clients = pgTable("clients", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  displayName: text("display_name").notNull(),
  legalName: text("legal_name").notNull(),
  pan: text("pan").notNull().unique(),
  aadhaarLast4: text("aadhaar_last4"),
  dateOfBirth: timestamp("date_of_birth"),
  incorporationDate: timestamp("incorporation_date"),
  address: jsonb("address"),
  contacts: jsonb("contacts").default([]),
  riskCategory: text("risk_category").default("MEDIUM"),
  tags: jsonb("tags").default([]),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

export const engagements = pgTable("engagements", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  clientId: uuid("client_id").references(() => clients.id).notNull(),
  ay: text("ay").notNull(), // Assessment Year like "2024-25"
  itrFormType: itrFormTypeEnum("itr_form_type").notNull(),
  status: engagementStatusEnum("status").default("DATA_REQUESTED"),
  tier: tierEnum("tier").notNull(),
  isAuditApplicable: boolean("is_audit_applicable").default(false),
  dueDates: jsonb("due_dates").default({}),
  slaProfileId: uuid("sla_profile_id"),
  priority: priorityEnum("priority").default("MEDIUM"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

export const assignments = pgTable("assignments", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  engagementId: uuid("engagement_id").references(() => engagements.id).notNull(),
  userId: uuid("user_id").references(() => users.id).notNull(),
  roleOnCase: text("role_on_case").notNull(), // "PREPARER", "REVIEWER", "PARTNER"
  startDate: timestamp("start_date").defaultNow(),
  endDate: timestamp("end_date"),
  createdAt: timestamp("created_at").defaultNow()
});

export const documents = pgTable("documents", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  engagementId: uuid("engagement_id").references(() => engagements.id).notNull(),
  docType: text("doc_type").notNull(),
  fileName: text("file_name").notNull(),
  filePath: text("file_path").notNull(),
  fileSize: integer("file_size").notNull(),
  checksum: text("checksum").notNull(),
  uploadedBy: uuid("uploaded_by").references(() => users.id).notNull(),
  clientVisible: boolean("client_visible").default(false),
  version: integer("version").default(1),
  createdAt: timestamp("created_at").defaultNow()
});

export const checklists = pgTable("checklists", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  engagementId: uuid("engagement_id").references(() => engagements.id).notNull(),
  name: text("name").notNull(),
  version: text("version").default("1.0"),
  isTemplate: boolean("is_template").default(false),
  createdAt: timestamp("created_at").defaultNow()
});

export const checklistItems = pgTable("checklist_items", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  checklistId: uuid("checklist_id").references(() => checklists.id).notNull(),
  item: text("item").notNull(),
  sectionRef: text("section_ref"), // e.g., "Sch CG – 112A"
  mandatory: boolean("mandatory").default(false),
  status: checklistStatusEnum("status").default("PENDING"),
  assigneeId: uuid("assignee_id").references(() => users.id),
  dueAt: timestamp("due_at"),
  completedAt: timestamp("completed_at"),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

export const computations = pgTable("computations", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  engagementId: uuid("engagement_id").references(() => engagements.id).notNull(),
  engine: text("engine").notNull(), // "excel", "python"
  version: text("version").default("1.0"),
  summaryJson: jsonb("summary_json").notNull(),
  attachments: jsonb("attachments").default([]),
  preparedBy: uuid("prepared_by").references(() => users.id).notNull(),
  reviewedBy: uuid("reviewed_by").references(() => users.id),
  reviewStatus: reviewStatusEnum("review_status").default("PENDING"),
  remarks: text("remarks"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

export const efiling = pgTable("efiling", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  engagementId: uuid("engagement_id").references(() => engagements.id).notNull(),
  jsonUploaded: boolean("json_uploaded").default(false),
  ackNo: text("ack_no"),
  filedAt: timestamp("filed_at"),
  eVerifyStatus: eVerifyStatusEnum("e_verify_status").default("PENDING"),
  eVerifiedAt: timestamp("e_verified_at"),
  cpcStatus: cpcStatusEnum("cpc_status"),
  refundStatus: refundStatusEnum("refund_status"),
  refundAmount: decimal("refund_amount"),
  cpcCommRef: text("cpc_comm_ref"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

export const comments = pgTable("comments", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  engagementId: uuid("engagement_id").references(() => engagements.id).notNull(),
  entityType: text("entity_type").notNull(),
  entityId: uuid("entity_id").notNull(),
  authorId: uuid("author_id").references(() => users.id).notNull(),
  body: text("body").notNull(),
  mentions: jsonb("mentions").default([]),
  createdAt: timestamp("created_at").defaultNow()
});

export const reminders = pgTable("reminders", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  engagementId: uuid("engagement_id").references(() => engagements.id).notNull(),
  channel: reminderChannelEnum("channel").notNull(),
  templateKey: text("template_key").notNull(),
  scheduleAt: timestamp("schedule_at").notNull(),
  sentAt: timestamp("sent_at"),
  status: reminderStatusEnum("status").default("SCHEDULED"),
  payloadJson: jsonb("payload_json").notNull(),
  createdAt: timestamp("created_at").defaultNow()
});

export const slaProfiles = pgTable("sla_profiles", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  stageTargetsJson: jsonb("stage_targets_json").notNull(),
  createdAt: timestamp("created_at").defaultNow()
});

export const slaEvents = pgTable("sla_events", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  engagementId: uuid("engagement_id").references(() => engagements.id).notNull(),
  stage: text("stage").notNull(),
  breached: boolean("breached").default(false),
  breachedAt: timestamp("breached_at"),
  createdAt: timestamp("created_at").defaultNow()
});

export const activityLog = pgTable("activity_log", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  actorUserId: uuid("actor_user_id").references(() => users.id),
  actorRole: text("actor_role").notNull(),
  action: text("action").notNull(),
  entityType: text("entity_type").notNull(),
  entityId: uuid("entity_id").notNull(),
  ip: text("ip").notNull(),
  userAgent: text("user_agent"),
  beforeJson: jsonb("before_json"),
  afterJson: jsonb("after_json"),
  signatureHash: text("signature_hash").notNull(),
  createdAt: timestamp("created_at").defaultNow()
});

export const statusTransitions = pgTable("status_transitions", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  engagementId: uuid("engagement_id").references(() => engagements.id).notNull(),
  fromStatus: text("from_status"),
  toStatus: text("to_status").notNull(),
  reason: text("reason"),
  actorUserId: uuid("actor_user_id").references(() => users.id).notNull(),
  createdAt: timestamp("created_at").defaultNow()
});

export const tags = pgTable("tags", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull().unique(),
  color: text("color").default("#6B7280"),
  createdAt: timestamp("created_at").defaultNow()
});

export const entityTags = pgTable("entity_tags", {
  id: uuid("id").primaryKey().default(sql`gen_random_uuid()`),
  entityType: text("entity_type").notNull(),
  entityId: uuid("entity_id").notNull(),
  tagId: uuid("tag_id").references(() => tags.id).notNull(),
  createdAt: timestamp("created_at").defaultNow()
});

// Insert schemas
export const insertUserSchema = createInsertSchema(users).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  lastLoginAt: true
});

export const insertClientSchema = createInsertSchema(clients).omit({
  id: true,
  createdAt: true,
  updatedAt: true
});

export const insertEngagementSchema = createInsertSchema(engagements).omit({
  id: true,
  createdAt: true,
  updatedAt: true
});

export const insertDocumentSchema = createInsertSchema(documents).omit({
  id: true,
  createdAt: true
});

export const insertCommentSchema = createInsertSchema(comments).omit({
  id: true,
  createdAt: true
});

export const insertChecklistItemSchema = createInsertSchema(checklistItems).omit({
  id: true,
  createdAt: true,
  updatedAt: true
});

export const insertComputationSchema = createInsertSchema(computations).omit({
  id: true,
  createdAt: true,
  updatedAt: true
});

// Types
export type User = typeof users.$inferSelect;
export type InsertUser = z.infer<typeof insertUserSchema>;
export type Client = typeof clients.$inferSelect;
export type InsertClient = z.infer<typeof insertClientSchema>;
export type Engagement = typeof engagements.$inferSelect;
export type InsertEngagement = z.infer<typeof insertEngagementSchema>;
export type Document = typeof documents.$inferSelect;
export type InsertDocument = z.infer<typeof insertDocumentSchema>;
export type Comment = typeof comments.$inferSelect;
export type InsertComment = z.infer<typeof insertCommentSchema>;
export type ChecklistItem = typeof checklistItems.$inferSelect;
export type InsertChecklistItem = z.infer<typeof insertChecklistItemSchema>;
export type Computation = typeof computations.$inferSelect;
export type InsertComputation = z.infer<typeof insertComputationSchema>;
export type ActivityLogEntry = typeof activityLog.$inferSelect;

// Login schema
export const loginSchema = z.object({
  email: z.string().email(),
  password: z.string().min(6)
});

export type LoginCredentials = z.infer<typeof loginSchema>;
