import React, { useState, useCallback, useEffect } from 'react';
import type { AppView, AvatarProfile, Garment, WearEvent } from './types';
import { GarmentCategory } from './types';
import BottomNav from './components/layout/BottomNav';
import AvatarCreator from './components/onboarding/AvatarCreator';
import AddItemForm from './components/wardrobe/AddItemForm';
import VirtualTryOn from './components/tryon/VirtualTryOn';
import ReportsDashboard from './components/reports/ReportsDashboard';
import LogWearEventForm from './components/wear-history/LogWearEventForm';
import TravelPlanner from './components/travel/TravelPlanner';
import { PlusIcon, WardrobeIcon, BackIcon } from './components/common/Icons';

// Mock Data
const MOCK_GARMENTS: Garment[] = [
    { id: '1', name: 'Classic White Tee', category: GarmentCategory.TOP, brand: 'Everlane', size: 'M', color: 'White', price: 2500, purchaseDate: '2023-05-10', tags: ['casual', 'basic', 'cotton'], frontImageUrl: 'https://placehold.co/400x400/FFF/333?text=White+Tee', careInstructions: 'Machine wash cold.' },
    { id: '2', name: 'High-Waisted Jeans', category: GarmentCategory.BOTTOM, brand: 'Levi\'s', size: '28', color: 'Blue', price: 8000, purchaseDate: '2023-03-15', tags: ['denim', 'casual'], frontImageUrl: 'https://placehold.co/400x400/557ac6/FFF?text=Jeans', careInstructions: 'Wash inside out.' },
    { id: '3', name: 'Floral Sundress', category: GarmentCategory.DRESS, brand: 'Reformation', size: '6', color: 'Yellow', price: 15000, purchaseDate: '2023-06-01', tags: ['summer', 'floral'], frontImageUrl: 'https://placehold.co/400x600/f5d547/333?text=Sundress', careInstructions: 'Dry clean only.' },
    { id: '4', name: 'Beige Trench Coat', category: GarmentCategory.OUTERWEAR, brand: 'Burberry', size: 'S', color: 'Beige', price: 150000, purchaseDate: '2022-10-20', tags: ['classic', 'investment'], frontImageUrl: 'https://placehold.co/400x600/D2B48C/FFF?text=Trench', careInstructions: 'Dry clean only.' }
];

const MOCK_WEAR_HISTORY: WearEvent[] = [
    { id: 'w1', garmentIds: ['1', '2'], date: '2023-08-01', occasion: 'Casual', notes: 'Coffee with friends' },
    { id: 'w2', garmentIds: ['3'], date: '2023-08-05', occasion: 'Date Night', notes: 'Dinner at that new Italian place' },
     { id: 'w3', garmentIds: ['1', '2'], date: '2023-08-10', occasion: 'Work', notes: 'A regular day at the office' },
];

const GarmentCard: React.FC<{ garment: Garment }> = ({ garment }) => (
    <div className="bg-white rounded-lg shadow-md overflow-hidden">
        <img src={garment.frontImageUrl} alt={garment.name} className="w-full h-48 object-cover"/>
        <div className="p-4">
            <h3 className="font-bold text-lg">{garment.name}</h3>
            <p className="text-sm text-stone-600">{garment.brand}</p>
            <div className="flex flex-wrap gap-1 mt-2">
                {garment.tags.map(tag => (
                    <span key={tag} className="bg-pink-100 text-pink-800 text-xs font-medium px-2 py-0.5 rounded-full">{tag}</span>
                ))}
            </div>
        </div>
    </div>
);


const WardrobeView: React.FC<{ wardrobe: Garment[]; setView: (view: AppView) => void }> = ({ wardrobe, setView }) => (
    <div className="p-4 pb-24">
        <div className="flex items-center mb-6">
            <button onClick={() => setView('DASHBOARD')} className="p-2 rounded-full hover:bg-stone-200 mr-2">
                <BackIcon className="h-6 w-6"/>
            </button>
            <h1 className="text-3xl font-bold">My Wardrobe</h1>
        </div>
        {wardrobe.length > 0 ? (
            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                {wardrobe.map(g => <GarmentCard key={g.id} garment={g}/>)}
            </div>
        ) : (
            <div className="text-center py-10 flex flex-col items-center justify-center h-[calc(100vh-200px)]">
                <WardrobeIcon className="w-24 h-24 text-stone-300 mb-4" />
                <h2 className="text-2xl font-semibold text-stone-700 mb-2">Your wardrobe is waiting!</h2>
                <p className="text-stone-500 mb-6 max-w-xs mx-auto">Start building your virtual closet by adding your first piece of clothing.</p>
                <button
                    onClick={() => setView('ADD_ITEM')}
                    className="bg-pink-600 text-white font-bold py-3 px-6 rounded-lg hover:bg-pink-700 transition-all flex items-center gap-2 shadow-lg transform hover:scale-105"
                >
                    <PlusIcon className="w-5 h-5" />
                    Add Your First Item
                </button>
            </div>
        )}
    </div>
);

const getFromStorage = <T,>(key: string, defaultValue: T): T => {
    try {
        const item = window.localStorage.getItem(key);
        return item ? JSON.parse(item) : defaultValue;
    } catch (error) {
        console.warn(`Error reading localStorage key "${key}":`, error);
        return defaultValue;
    }
};

const App: React.FC = () => {
  const [avatar, setAvatar] = useState<AvatarProfile | null>(() => getFromStorage('tryvatar_avatar', null));
  const [view, setView] = useState<AppView>(() => (getFromStorage('tryvatar_avatar', null) ? 'DASHBOARD' : 'ONBOARDING'));
  const [wardrobe, setWardrobe] = useState<Garment[]>(() => getFromStorage('tryvatar_wardrobe', []));
  const [wearHistory, setWearHistory] = useState<WearEvent[]>(() => getFromStorage('tryvatar_wearHistory', []));
  const [addItemBackView, setAddItemBackView] = useState<AppView>('DASHBOARD');

  useEffect(() => {
    try {
        if (avatar) {
            localStorage.setItem('tryvatar_avatar', JSON.stringify(avatar));
        } else {
            localStorage.removeItem('tryvatar_avatar');
        }
    } catch (error) {
        console.warn('Error saving avatar to localStorage:', error);
    }
  }, [avatar]);

  useEffect(() => {
    try {
        localStorage.setItem('tryvatar_wardrobe', JSON.stringify(wardrobe));
    } catch (error) {
        console.warn('Error saving wardrobe to localStorage:', error);
    }
  }, [wardrobe]);

  useEffect(() => {
    try {
        localStorage.setItem('tryvatar_wearHistory', JSON.stringify(wearHistory));
    } catch (error) {
        console.warn('Error saving wear history to localStorage:', error);
    }
  }, [wearHistory]);

  const appSetView = useCallback((targetView: AppView) => {
    if (targetView === 'ADD_ITEM') {
      // Don't set back view if we are already in ADD_ITEM or coming from ONBOARDING
      if (view !== 'ADD_ITEM' && view !== 'ONBOARDING') {
        setAddItemBackView(view);
      }
    }
    setView(targetView);
  }, [view]);


  const handleOnboardingComplete = useCallback((newAvatar: AvatarProfile) => {
    setAvatar(newAvatar);
    // Only populate with mock data if the wardrobe has never been touched.
    if (localStorage.getItem('tryvatar_wardrobe') === null) {
      setWardrobe(MOCK_GARMENTS);
      setWearHistory(MOCK_WEAR_HISTORY);
    }
    setView('DASHBOARD');
  }, []);
  
  const handleAddItem = useCallback((newItem: Garment) => {
    setWardrobe(prev => [newItem, ...prev]);
  }, []);

  const handleAddWearEvent = useCallback((newEventData: Omit<WearEvent, 'id'>) => {
    const newEvent: WearEvent = {
        id: `w${Date.now()}`,
        ...newEventData
    };
    setWearHistory(prev => [newEvent, ...prev]);
    setView('REPORTS'); // Go back to reports page after logging
  }, []);

  const renderView = () => {
    if (!avatar || view === 'ONBOARDING') {
      return <AvatarCreator onOnboardingComplete={handleOnboardingComplete} />;
    }

    switch (view) {
      case 'DASHBOARD':
        return <ReportsDashboard wardrobe={wardrobe} wearHistory={wearHistory} setView={appSetView}/>;
      case 'WARDROBE':
        return <WardrobeView wardrobe={wardrobe} setView={appSetView} />;
      case 'ADD_ITEM':
        return <AddItemForm onAddItem={handleAddItem} onBack={() => appSetView(addItemBackView)} />;
      case 'TRY_ON':
        return <VirtualTryOn wardrobe={wardrobe} avatar={avatar} onBack={() => setView('DASHBOARD')} />;
      case 'REPORTS':
        return <ReportsDashboard wardrobe={wardrobe} wearHistory={wearHistory} setView={appSetView} />;
      case 'LOG_WEAR':
        return <LogWearEventForm wardrobe={wardrobe} onLogEvent={handleAddWearEvent} onBack={() => setView('REPORTS')} setView={appSetView} />;
      case 'TRAVEL_PLANNER':
        return <TravelPlanner wardrobe={wardrobe} onBack={() => setView('DASHBOARD')} />;
      default:
        return <div>Dashboard</div>;
    }
  };

  return (
    <div className="font-sans antialiased">
      <main className="min-h-screen">
          {renderView()}
      </main>
      {avatar && view !== 'ONBOARDING' && <BottomNav activeView={view} setView={appSetView} />}
    </div>
  );
};

export default App;