import React, { useState, useRef, useEffect, useCallback } from 'react';
import type { AvatarProfile } from '../../types';
import { CameraIcon } from '../common/Icons';
import LoadingSpinner from '../common/LoadingSpinner';

interface AvatarCreatorProps {
  onOnboardingComplete: (avatar: AvatarProfile) => void;
}

type OnboardingStep = 'welcome' | 'frontPhoto' | 'sidePhoto' | 'processing' | 'complete';

const AvatarCreator: React.FC<AvatarCreatorProps> = ({ onOnboardingComplete }) => {
  const [step, setStep] = useState<OnboardingStep>('welcome');
  const [frontPhoto, setFrontPhoto] = useState<string | null>(null);
  const [sidePhoto, setSidePhoto] = useState<string | null>(null);
  const [cameraError, setCameraError] = useState<string | null>(null);
  const [isCameraLoading, setIsCameraLoading] = useState(false);
  const [isVideoPlaying, setIsVideoPlaying] = useState(false);
  
  const fileInputRef = useRef<HTMLInputElement>(null);
  const videoRef = useRef<HTMLVideoElement>(null);
  const canvasRef = useRef<HTMLCanvasElement>(null);
  const streamRef = useRef<MediaStream | null>(null);

  const startCamera = useCallback(async () => {
    if (streamRef.current) return; // Already running
    
    setIsCameraLoading(true);
    setCameraError(null);
    try {
        const stream = await navigator.mediaDevices.getUserMedia({ video: { facingMode: 'user' } });
        streamRef.current = stream;
        if (videoRef.current) {
            videoRef.current.srcObject = stream;
        }
    } catch (err) {
        console.error("Error accessing camera:", err);
        setCameraError("Could not access your camera. Please check browser permissions or upload a file instead.");
    } finally {
        setIsCameraLoading(false);
    }
  }, []);

  const stopCamera = useCallback(() => {
      if (streamRef.current) {
          streamRef.current.getTracks().forEach(track => track.stop());
          streamRef.current = null;
      }
      if (videoRef.current) {
        videoRef.current.srcObject = null;
      }
  }, []);

  useEffect(() => {
    const wantsCamera = step === 'frontPhoto' || step === 'sidePhoto';

    if (wantsCamera && !streamRef.current) {
        setIsVideoPlaying(false);
        startCamera();
    } else if (!wantsCamera && streamRef.current) {
        stopCamera();
    }
    
    // Cleanup on component unmount
    return () => {
      if (streamRef.current) {
        stopCamera();
      }
    };
  }, [step, startCamera, stopCamera]);


  const handlePhotoData = (dataUrl: string) => {
    if (step === 'frontPhoto') {
      setFrontPhoto(dataUrl);
      setStep('sidePhoto');
    } else if (step === 'sidePhoto') {
      setSidePhoto(dataUrl);
      setStep('processing');
      // Simulate processing time
      setTimeout(() => {
        if (!frontPhoto) {
          console.error("Critical error: frontPhoto is missing when creating avatar.");
          alert("An error occurred while creating your avatar. Please start over.");
          setStep('frontPhoto'); // Reset to the beginning
          return;
        }
        const newAvatar: AvatarProfile = {
          frontPhoto: frontPhoto,
          sidePhoto: dataUrl,
          avatarUrl: 'https://picsum.photos/seed/avatar/400/600',
        };
        onOnboardingComplete(newAvatar);
        setStep('complete');
      }, 3000);
    }
  };

  const capturePhoto = () => {
    if (videoRef.current && canvasRef.current && videoRef.current.readyState >= 3) { // Use readyState 3 or 4
      const video = videoRef.current;
      const canvas = canvasRef.current;
      canvas.width = video.videoWidth;
      canvas.height = video.videoHeight;
      const context = canvas.getContext('2d');
      if (context) {
        // Flip the image horizontally to counteract the mirror effect of the video preview
        context.translate(canvas.width, 0);
        context.scale(-1, 1);
        context.drawImage(video, 0, 0, canvas.width, canvas.height);
        const dataUrl = canvas.toDataURL('image/jpeg');
        handlePhotoData(dataUrl);
      }
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (event) => {
        const result = event.target?.result as string;
        handlePhotoData(result);
      };
      reader.readAsDataURL(file);
    }
  };

  const renderStep = () => {
    switch (step) {
      case 'welcome':
        return (
          <div className="text-center">
              <h1 className="text-3xl font-bold mb-4">Welcome to Tryvatar</h1>
              <p className="text-stone-600 mb-8 max-w-sm mx-auto">
                  To get started, we'll help you create a personal avatar. This allows for a realistic virtual try-on experience. Your photos are processed securely on your device and are never uploaded to a server.
              </p>
              <button
                  onClick={() => setStep('frontPhoto')}
                  className="w-full bg-pink-600 text-white font-bold py-3 px-4 rounded-lg hover:bg-pink-700 transition-all shadow-md"
              >
                  Create My Avatar
              </button>
          </div>
        );
      case 'frontPhoto':
      case 'sidePhoto':
        const isFrontPhoto = step === 'frontPhoto';
        const title = isFrontPhoto ? 'Front-Facing Selfie' : 'Side-Profile Photo';
        const description = isFrontPhoto 
            ? 'Position yourself in the frame with arms relaxed at your sides. Use good lighting against a plain background.' 
            : 'Great! Now turn 90 degrees to either side. Keep your posture natural.';
        
        const commonHeader = (
             <div className="text-center">
                <h2 className="text-2xl font-bold mb-2">{title}</h2>
                <p className="text-stone-600 mb-4">{description}</p>
                {isFrontPhoto && <p className="text-xs text-stone-500 mb-6">Your photos stay on your device to create your private avatar.</p>}
            </div>
        );

        if (isCameraLoading) {
            return (
                <div>
                    {commonHeader}
                    <div className="w-full rounded-lg mb-4 aspect-[4/3] bg-stone-200 flex items-center justify-center">
                         <LoadingSpinner text="Starting camera..."/>
                    </div>
                </div>
            );
        }

        if (cameraError) {
            return (
                <div>
                    {commonHeader}
                    <div className="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg relative mb-4" role="alert">
                        <span className="block sm:inline">{cameraError}</span>
                    </div>
                    <div className="space-y-4">
                        <button
                            onClick={startCamera}
                            className="w-full flex items-center justify-center gap-2 bg-white text-stone-800 font-bold py-3 px-4 rounded-lg hover:bg-stone-50 transition-all border-2 border-stone-300"
                        >
                            <CameraIcon className="w-6 h-6"/>
                            Try Webcam Again
                        </button>
                        <div className="relative flex py-2 items-center">
                            <div className="flex-grow border-t border-stone-300"></div>
                            <span className="flex-shrink mx-4 text-stone-500">OR</span>
                            <div className="flex-grow border-t border-stone-300"></div>
                        </div>
                        <button
                            onClick={() => fileInputRef.current?.click()} 
                            className="w-full bg-stone-100 text-stone-700 font-bold py-3 px-4 rounded-lg hover:bg-stone-200 transition-all"
                        >
                            Upload from Device
                        </button>
                    </div>
                    <input
                        ref={fileInputRef}
                        type="file"
                        accept="image/*"
                        onChange={handleFileChange}
                        className="hidden"
                    />
                </div>
            );
        }
        
        // If no error and not loading, show camera view.
        return (
            <div>
                {commonHeader}
                <div className="relative w-full rounded-lg mb-4 aspect-[4/3] bg-stone-200">
                    <video
                        ref={videoRef}
                        playsInline
                        autoPlay
                        muted
                        onCanPlay={() => setIsVideoPlaying(true)}
                        className={`w-full h-full object-cover rounded-lg transform -scale-x-100 ${!isVideoPlaying ? 'invisible' : ''}`}
                    />
                    {!isVideoPlaying && (
                        <div className="absolute inset-0 flex items-center justify-center">
                            <LoadingSpinner text="Preparing camera..." />
                        </div>
                    )}
                </div>
                <div className="flex space-x-4">
                    <button
                        onClick={capturePhoto}
                        disabled={!isVideoPlaying}
                        className="w-full bg-pink-600 text-white font-bold py-3 px-4 rounded-lg hover:bg-pink-700 transition-all disabled:bg-stone-400 disabled:cursor-not-allowed"
                    >
                        Capture
                    </button>
                </div>
                <div className="relative flex py-4 items-center">
                    <div className="flex-grow border-t border-stone-300"></div>
                    <span className="flex-shrink mx-4 text-stone-500 text-sm">OR</span>
                    <div className="flex-grow border-t border-stone-300"></div>
                </div>
                 <button
                    onClick={() => fileInputRef.current?.click()} 
                    className="w-full bg-stone-100 text-stone-700 font-bold py-3 px-4 rounded-lg hover:bg-stone-200 transition-all"
                >
                    Upload from Device
                </button>
                <canvas ref={canvasRef} className="hidden" />
                 <input
                    ref={fileInputRef}
                    type="file"
                    accept="image/*"
                    onChange={handleFileChange}
                    className="hidden"
                 />
            </div>
        );

      case 'processing':
        return (
          <div className="text-center">
            <h2 className="text-2xl font-bold mb-2">Building Your Avatar</h2>
            <p className="text-stone-600 mb-8">This is where the magic happens. Your avatar is being created securely on your device.</p>
            <LoadingSpinner text="Analyzing..." />
          </div>
        );
        
      default:
        return null;
    }
  };

  return (
    <div className="min-h-screen bg-white flex items-center justify-center p-4">
        <div className="w-full max-w-md mx-auto">
            {renderStep()}
        </div>
    </div>
  );
};

export default AvatarCreator;