import React, { useState, useMemo } from 'react';
import type { Garment, WearEvent, AppView } from '../../types';
import { suggestNextPurchase, suggestOutfitFromWardrobe } from '../../services/geminiService';
import LoadingSpinner from '../common/LoadingSpinner';
import { PieChart, Pie, Cell, Tooltip, Legend, ResponsiveContainer } from 'recharts';
import { TryOnIcon, CheckCircleIcon, PlusIcon } from '../common/Icons';

interface ReportsDashboardProps {
  wardrobe: Garment[];
  wearHistory: WearEvent[];
  setView: (view: AppView) => void;
}

const Card: React.FC<{ title: string; children: React.ReactNode; className?: string }> = ({ title, children, className }) => (
  <div className={`bg-white p-4 rounded-xl shadow-md ${className}`}>
    <h3 className="font-bold text-lg mb-3 text-stone-800">{title}</h3>
    {children}
  </div>
);

const WearHistoryItem: React.FC<{ event: WearEvent, wardrobe: Garment[] }> = ({ event, wardrobe }) => {
    const garmentsInEvent = event.garmentIds.map(id => wardrobe.find(g => g.id === id)).filter(Boolean) as Garment[];

    return (
        <div className="flex items-start space-x-4 py-3 border-b border-stone-200 last:border-b-0">
            <div className="text-center w-16 flex-shrink-0">
                <p className="font-bold text-stone-800">{new Date(event.date).toLocaleDateString('en-US', { day: 'numeric', timeZone: 'UTC' })}</p>
                <p className="text-xs text-stone-500">{new Date(event.date).toLocaleDateString('en-US', { month: 'short', timeZone: 'UTC' })}</p>
            </div>
            <div className="flex-grow">
                <p className="font-semibold">{event.occasion}</p>
                {event.notes && <p className="text-sm text-stone-600 italic">"{event.notes}"</p>}
                <div className="flex flex-wrap gap-2 mt-2">
                    {garmentsInEvent.map(g => (
                        <div key={g.id} className="flex items-center space-x-2 bg-stone-100 rounded-full pr-2">
                           <img src={g.frontImageUrl} alt={g.name} className="w-8 h-8 rounded-full object-cover"/>
                           <span className="text-xs text-stone-700">{g.name}</span>
                        </div>
                    ))}
                </div>
            </div>
        </div>
    );
};


const ReportsDashboard: React.FC<ReportsDashboardProps> = ({ wardrobe, wearHistory, setView }) => {
  const [nextPurchaseSuggestion, setNextPurchaseSuggestion] = useState<{ suggestion: string; reason: string } | null>(null);
  const [isLoadingNextPurchase, setIsLoadingNextPurchase] = useState(false);
  
  const [outfitSuggestion, setOutfitSuggestion] = useState<{ outfitItems: string[]; reason: string } | null>(null);
  const [isLoadingOutfit, setIsLoadingOutfit] = useState(false);

  const getNextPurchaseSuggestion = async () => {
    setIsLoadingNextPurchase(true);
    const wardrobeDetails = wardrobe.map(g => `${g.name} (${g.category}, ${g.color})`);
    const result = await suggestNextPurchase(wardrobeDetails);
    setNextPurchaseSuggestion(result);
    setIsLoadingNextPurchase(false);
  };
  
  const getOutfitSuggestion = async () => {
    setIsLoadingOutfit(true);
    const wardrobeNames = wardrobe.map(g => g.name);
    const result = await suggestOutfitFromWardrobe(wardrobeNames);
    setOutfitSuggestion(result);
    setIsLoadingOutfit(false);
  };

  const costPerWear = useMemo(() => {
    return wardrobe.map(garment => {
      const wearCount = wearHistory.filter(e => e.garmentIds.includes(garment.id)).length;
      return {
        name: garment.name,
        costPerWear: wearCount > 0 ? (garment.price / wearCount).toFixed(2) : 'N/A',
        wears: wearCount
      };
    }).sort((a, b) => {
        if (a.costPerWear === 'N/A') return 1;
        if (b.costPerWear === 'N/A') return -1;
        return Number(a.costPerWear) - Number(b.costPerWear);
    }).slice(0, 5);
  }, [wardrobe, wearHistory]);

  const wearFrequency = useMemo(() => {
    return wardrobe.map(garment => {
      const wearCount = wearHistory.filter(event => event.garmentIds.includes(garment.id)).length;
      return {
        name: garment.name,
        wearCount,
      };
    })
    .filter(item => item.wearCount > 0)
    .sort((a, b) => b.wearCount - a.wearCount)
    .slice(0, 5);
  }, [wardrobe, wearHistory]);

  const categorySpend = useMemo(() => {
    const spendMap = new Map<string, number>();
    wardrobe.forEach(g => {
        spendMap.set(g.category, (spendMap.get(g.category) || 0) + g.price);
    });
    return Array.from(spendMap.entries()).map(([name, value]) => ({ name, value }));
  }, [wardrobe]);
  
  const sortedWearHistory = useMemo(() => {
    return [...wearHistory].sort((a, b) => new Date(b.date).getTime() - new Date(a.date).getTime());
  }, [wearHistory]);

  const COLORS = ['#FFC0CB', '#FFB6C1', '#FF69B4', '#FF1493', '#DB7093', '#C71585'];

  return (
    <div className="p-4 bg-stone-50 min-h-screen pb-24 space-y-6">
      <h1 className="text-3xl font-bold text-stone-900">My Wardrobe Insights</h1>

      <Card title="Virtual Try-On" className="bg-pink-50 border border-pink-200">
        <div className="flex flex-col items-center text-center">
            <TryOnIcon className="w-16 h-16 text-pink-500 mb-4" />
            <p className="text-stone-600 mb-4 max-w-sm">
                See how your clothes look on you. Mix and match items from your wardrobe on your virtual avatar.
            </p>
            <button
                onClick={() => setView('TRY_ON')}
                className="bg-pink-600 text-white font-bold py-2 px-6 rounded-lg hover:bg-pink-700 transition-all shadow-md"
            >
                Try an Outfit
            </button>
        </div>
      </Card>

      <Card title="Quick Actions">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="bg-pink-50 rounded-lg p-4 flex flex-col items-center text-center">
                <h4 className="font-semibold text-lg mb-2">Log an Outfit</h4>
                <p className="text-stone-600 text-sm mb-4 flex-grow">Track what you wear to get personalized style insights.</p>
                <button
                    onClick={() => setView('LOG_WEAR')}
                    className="w-full bg-pink-600 text-white font-bold py-2 px-5 rounded-lg hover:bg-pink-700 transition-all flex items-center justify-center gap-2"
                >
                    <CheckCircleIcon className="w-5 h-5" />
                    Log Today's Look
                </button>
            </div>
            <div className="bg-stone-100 rounded-lg p-4 flex flex-col items-center text-center">
                <h4 className="font-semibold text-lg mb-2">Add New Item</h4>
                <p className="text-stone-600 text-sm mb-4 flex-grow">Expand your virtual closet with your latest purchase.</p>
                <button
                    onClick={() => setView('ADD_ITEM')}
                    className="w-full bg-white text-stone-800 font-bold py-2 px-5 rounded-lg hover:bg-stone-200 transition-all border border-stone-300 flex items-center justify-center gap-2"
                >
                    <PlusIcon className="w-5 h-5" />
                    Add a Purchase
                </button>
            </div>
        </div>
      </Card>

      <Card title="Outfit Log">
        <div className="mb-4">
            {sortedWearHistory.length > 0 ? (
                <div>
                    {sortedWearHistory.slice(0, 5).map(event => (
                        <WearHistoryItem key={event.id} event={event} wardrobe={wardrobe} />
                    ))}
                </div>
            ) : (
                <p className="text-stone-600 text-center py-4">No outfits logged yet. Start tracking your looks!</p>
            )}
        </div>
      </Card>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card title="What to Wear Next">
          {outfitSuggestion ? (
            <div>
              <p className="text-stone-600 mb-3">{outfitSuggestion.reason}</p>
              <div className="flex flex-wrap gap-2 mb-3">
                {outfitSuggestion.outfitItems.map(itemName => {
                  const garment = wardrobe.find(g => g.name === itemName);
                  return garment ? (
                    <div key={garment.id} className="text-center flex-shrink-0">
                      <img src={garment.frontImageUrl} alt={garment.name} className="w-16 h-16 rounded-md object-cover shadow-sm" />
                      <p className="text-xs mt-1 max-w-[64px] truncate">{garment.name}</p>
                    </div>
                  ) : null;
                })}
              </div>
              <button
                onClick={getOutfitSuggestion}
                disabled={isLoadingOutfit}
                className="mt-4 bg-white text-pink-600 font-semibold py-2 px-4 rounded-lg hover:bg-pink-50 transition-all border border-pink-600 text-sm"
              >
                {isLoadingOutfit ? 'Thinking...' : 'Suggest Another Outfit'}
              </button>
            </div>
          ) : (
            <p className="text-stone-600">Get a fresh outfit idea from your existing wardrobe.</p>
          )}
          {!outfitSuggestion && (
            <button
              onClick={getOutfitSuggestion}
              disabled={isLoadingOutfit}
              className="mt-4 bg-pink-600 text-white font-bold py-2 px-4 rounded-lg hover:bg-pink-700 transition-all disabled:bg-stone-400"
            >
              {isLoadingOutfit ? <LoadingSpinner /> : 'Suggest an Outfit'}
            </button>
          )}
        </Card>
        
        <Card title="Travel Outfit Planner">
          <p className="text-stone-600">Get personalized outfit suggestions for your next trip based on your destination and wardrobe.</p>
          <button 
              onClick={() => setView('TRAVEL_PLANNER')} 
              className="mt-4 bg-pink-600 text-white font-bold py-2 px-4 rounded-lg hover:bg-pink-700 transition-all"
          >
            Plan My Trip
          </button>
        </Card>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card title="Best Value Items (Cost Per Wear)">
          <ul className="space-y-2 text-sm">
            {costPerWear.map(item => (
                <li key={item.name} className="flex justify-between">
                    <span className="truncate pr-2">{item.name}</span>
                    <span className="font-semibold text-pink-600">
                        {item.costPerWear !== 'N/A' ? `₹${item.costPerWear}` : 'Not Worn'}
                    </span>
                </li>
            ))}
          </ul>
        </Card>

        <Card title="Wear Frequency (Top 5)">
          {wearFrequency.length > 0 ? (
            <ul className="space-y-2 text-sm">
              {wearFrequency.map(item => (
                  <li key={item.name} className="flex justify-between">
                      <span className="truncate pr-2">{item.name}</span>
                      <span className="font-semibold text-pink-600">
                          {item.wearCount} {item.wearCount === 1 ? 'wear' : 'wears'}
                      </span>
                  </li>
              ))}
            </ul>
          ) : (
             <p className="text-stone-500 text-sm">No wear history logged yet.</p>
          )}
        </Card>

        <Card title="What to Purchase Next">
          {nextPurchaseSuggestion ? (
            <div>
              <p className="font-semibold text-pink-600 text-lg">{nextPurchaseSuggestion.suggestion}</p>
              <p className="text-stone-600 mt-1">{nextPurchaseSuggestion.reason}</p>
              <button 
                  onClick={getNextPurchaseSuggestion} 
                  disabled={isLoadingNextPurchase}
                  className="mt-4 bg-white text-pink-600 font-semibold py-2 px-4 rounded-lg hover:bg-pink-50 transition-all border border-pink-600 text-sm"
              >
                {isLoadingNextPurchase ? 'Getting another...' : 'Suggest Another'}
              </button>
            </div>
          ) : (
            <p className="text-stone-600">Find out what key piece could unlock more outfits in your current wardrobe.</p>
          )}
          {!nextPurchaseSuggestion && (<button 
              onClick={getNextPurchaseSuggestion} 
              disabled={isLoadingNextPurchase}
              className="mt-4 bg-pink-600 text-white font-bold py-2 px-4 rounded-lg hover:bg-pink-700 transition-all disabled:bg-stone-400"
          >
            {isLoadingNextPurchase ? <LoadingSpinner /> : 'Find My Next Piece'}
          </button>)}
        </Card>

        <Card title="Neglected Items (Not Worn >90 Days)">
            <p className="text-stone-500 text-sm">Feature coming soon!</p>
        </Card>
      </div>

      <Card title="Spend by Category">
        <div style={{ width: '100%', height: 300 }}>
            <ResponsiveContainer>
                <PieChart>
                    <Pie data={categorySpend} dataKey="value" nameKey="name" cx="50%" cy="50%" outerRadius={100} fill="#8884d8" label>
                        {categorySpend.map((entry, index) => <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />)}
                    </Pie>
                    <Tooltip formatter={(value: number) => `₹${value.toFixed(2)}`}/>
                    <Legend />
                </PieChart>
            </ResponsiveContainer>
        </div>
      </Card>
      
    </div>
  );
};

export default ReportsDashboard;