import React, { useState } from 'react';
import type { Garment } from '../../types';
import { planTravelOutfits } from '../../services/geminiService';
import type { TravelOutfitPlan } from '../../services/geminiService';
import { BackIcon } from '../common/Icons';
import LoadingSpinner from '../common/LoadingSpinner';

interface TravelPlannerProps {
  wardrobe: Garment[];
  onBack: () => void;
}

const TravelPlanner: React.FC<TravelPlannerProps> = ({ wardrobe, onBack }) => {
  const [destination, setDestination] = useState('');
  const [duration, setDuration] = useState<number | ''>(5);
  const [travelType, setTravelType] = useState('Leisure');
  
  const [isLoading, setIsLoading] = useState(false);
  const [plan, setPlan] = useState<TravelOutfitPlan[] | null>(null);
  const [error, setError] = useState<string | null>(null);

  const handleGeneratePlan = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!destination || !duration) {
      setError('Please fill in all the fields.');
      return;
    }

    setIsLoading(true);
    setError(null);
    setPlan(null);

    const wardrobeNames = wardrobe.map(g => g.name);
    try {
      const result = await planTravelOutfits(destination, duration, travelType, wardrobeNames);
      if (result && result.length > 0) {
        setPlan(result);
      } else {
        setError("Could not generate a plan. Your wardrobe might be missing key items for this trip, or there was an issue with the AI. Please try again.");
      }
    } catch (err) {
      console.error(err);
      setError("An unexpected error occurred while creating your travel plan.");
    } finally {
      setIsLoading(false);
    }
  };

  const findGarmentImage = (name: string): string | undefined => {
      return wardrobe.find(g => g.name.toLowerCase() === name.toLowerCase())?.frontImageUrl;
  }

  const renderPlan = () => {
      if (!plan) return null;

      return (
          <div className="mt-8 space-y-6">
              <h2 className="text-2xl font-bold text-center">Your Outfit Plan for {destination}</h2>
              {plan.map(dayPlan => (
                  <div key={dayPlan.day} className="bg-white p-4 rounded-xl shadow-md">
                      <h3 className="font-bold text-lg text-pink-600">Day {dayPlan.day}: <span className="text-stone-800">{dayPlan.occasion}</span></h3>
                      <div className="flex flex-wrap gap-4 mt-3">
                          {dayPlan.outfitItems.map(itemName => {
                              const imageUrl = findGarmentImage(itemName);
                              return (
                                <div key={itemName} className="text-center">
                                    {imageUrl ? (
                                        <img src={imageUrl} alt={itemName} className="w-24 h-24 rounded-lg object-cover shadow-sm"/>
                                    ) : (
                                        <div className="w-24 h-24 rounded-lg bg-stone-200 flex items-center justify-center">
                                            <span className="text-xs text-stone-500 p-1">{itemName}</span>
                                        </div>
                                    )}
                                    <p className="text-xs mt-1 max-w-[96px]">{itemName}</p>
                                </div>
                              )
                          })}
                      </div>
                  </div>
              ))}
          </div>
      )
  };


  return (
    <div className="p-4 max-w-2xl mx-auto pb-24">
      <div className="flex items-center mb-6">
        <button onClick={onBack} className="p-2 rounded-full hover:bg-stone-200 mr-2">
          <BackIcon className="h-6 w-6" />
        </button>
        <h1 className="text-3xl font-bold">Travel Outfit Planner</h1>
      </div>
      
      {!plan && !isLoading && (
        <form onSubmit={handleGeneratePlan} className="space-y-6 bg-white p-6 rounded-xl shadow-md">
            <div>
                <label htmlFor="destination" className="block text-sm font-medium text-stone-700">Destination</label>
                <input type="text" id="destination" value={destination} onChange={(e) => setDestination(e.target.value)} required placeholder="e.g., Paris, France" className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"/>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label htmlFor="duration" className="block text-sm font-medium text-stone-700">Number of Days</label>
                    <input type="number" id="duration" value={duration} onChange={(e) => setDuration(e.target.value === '' ? '' : Number(e.target.value))} required min="1" className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"/>
                </div>
                <div>
                    <label htmlFor="travelType" className="block text-sm font-medium text-stone-700">Type of Travel</label>
                    <select id="travelType" value={travelType} onChange={(e) => setTravelType(e.target.value)} className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500">
                        <option>Leisure</option>
                        <option>Business</option>
                        <option>Adventure</option>
                        <option>Beach</option>
                        <option>City Exploration</option>
                    </select>
                </div>
            </div>

            {error && <p className="text-red-600 text-sm">{error}</p>}
            
            <button
                type="submit"
                className="w-full bg-pink-600 text-white font-bold py-3 px-4 rounded-lg hover:bg-pink-700 transition-all shadow-md"
            >
                Generate Outfit Plan
            </button>
        </form>
      )}

      {isLoading && <div className="flex justify-center mt-8"><LoadingSpinner text="Planning your outfits..." /></div>}

      {renderPlan()}
    </div>
  );
};

export default TravelPlanner;
