import React, { useState } from 'react';
import type { Garment, AvatarProfile } from '../../types';
import { GarmentCategory } from '../../types';
import { generateVirtualTryOnImage } from '../../services/geminiService';
import LoadingSpinner from '../common/LoadingSpinner';
import { BackIcon } from '../common/Icons';

interface VirtualTryOnProps {
  wardrobe: Garment[];
  avatar: AvatarProfile;
  onBack: () => void;
}

const GarmentItem: React.FC<{ garment: Garment, onClick: () => void, isSelected: boolean }> = ({ garment, onClick, isSelected }) => (
    <div onClick={onClick} className={`cursor-pointer rounded-lg overflow-hidden border-2 ${isSelected ? 'border-pink-500' : 'border-transparent'} transition-all`}>
        <img src={garment.frontImageUrl} alt={garment.name} className="w-full h-24 object-cover" />
    </div>
);


const VirtualTryOn: React.FC<VirtualTryOnProps> = ({ wardrobe, avatar, onBack }) => {
  const [selectedTop, setSelectedTop] = useState<Garment | null>(null);
  const [selectedBottom, setSelectedBottom] = useState<Garment | null>(null);
  const [selectedDress, setSelectedDress] = useState<Garment | null>(null);
  const [selectedOuterwear, setSelectedOuterwear] = useState<Garment | null>(null);
  const [generatedImage, setGeneratedImage] = useState<string | null>(null);
  const [isGenerating, setIsGenerating] = useState(false);

  const handleItemSelect = (item: Garment) => {
    // Reset generated image when selection changes
    if (generatedImage) setGeneratedImage(null);

    switch(item.category) {
        case GarmentCategory.TOP:
            setSelectedTop(prev => prev?.id === item.id ? null : item);
            setSelectedDress(null);
            break;
        case GarmentCategory.BOTTOM:
            setSelectedBottom(prev => prev?.id === item.id ? null : item);
            setSelectedDress(null);
            break;
        case GarmentCategory.DRESS:
            setSelectedDress(prev => prev?.id === item.id ? null : item);
            setSelectedTop(null);
            setSelectedBottom(null);
            break;
        case GarmentCategory.OUTERWEAR:
            setSelectedOuterwear(prev => prev?.id === item.id ? null : item);
            break;
    }
  }

  const handleGenerateTryOn = async () => {
    const selectedGarments = [selectedTop, selectedBottom, selectedDress, selectedOuterwear].filter(Boolean) as Garment[];
    if (selectedGarments.length === 0) return;

    setIsGenerating(true);
    setGeneratedImage(null);

    const garmentsForApi = selectedGarments.map(g => ({
        name: g.name,
        imageUrl: g.frontImageUrl,
    }));

    try {
      const result = await generateVirtualTryOnImage(avatar.frontPhoto, garmentsForApi);
      if (result) {
        setGeneratedImage(result);
      } else {
        alert("Sorry, we couldn't generate the try-on image. Please try again.");
      }
    } catch (error) {
        console.error("Try-on generation failed:", error);
        alert("An error occurred while generating the try-on image.");
    } finally {
        setIsGenerating(false);
    }
  };

  const renderCategory = (category: GarmentCategory, title: string) => {
    const items = wardrobe.filter(g => g.category === category);
    if (items.length === 0) return null;

    return (
        <div className="mb-4">
            <h3 className="font-semibold text-lg mb-2">{title}</h3>
            <div className="grid grid-cols-3 gap-2">
                {items.map(item => (
                    <GarmentItem 
                        key={item.id} 
                        garment={item} 
                        onClick={() => handleItemSelect(item)}
                        isSelected={
                            selectedTop?.id === item.id || 
                            selectedBottom?.id === item.id || 
                            selectedDress?.id === item.id ||
                            selectedOuterwear?.id === item.id
                        }
                    />
                ))}
            </div>
        </div>
    );
  }

  const selectedItemsCount = [selectedTop, selectedBottom, selectedDress, selectedOuterwear].filter(Boolean).length;

  return (
    <div className="flex flex-col md:flex-row h-screen bg-stone-100">
        {/* Left Side: Clothing Picker */}
        <div className="w-full md:w-1/3 lg:w-1/4 bg-white p-4 overflow-y-auto order-2 md:order-1 h-1/2 md:h-full">
            <div className="flex items-center mb-6">
                <button onClick={onBack} className="p-2 rounded-full hover:bg-stone-200 mr-2">
                    <BackIcon className="h-6 w-6"/>
                </button>
                <h2 className="text-2xl font-bold">My Closet</h2>
            </div>
            {renderCategory(GarmentCategory.DRESS, 'Dresses')}
            {renderCategory(GarmentCategory.TOP, 'Tops')}
            {renderCategory(GarmentCategory.BOTTOM, 'Bottoms')}
            {renderCategory(GarmentCategory.OUTERWEAR, 'Outerwear')}
        </div>

        {/* Right Side: Avatar Canvas */}
        <div className="w-full md:w-2/3 lg:w-3/4 flex-grow flex flex-col items-center justify-center p-4 order-1 md:order-2">
            <div className="relative w-[300px] h-[500px] bg-stone-200 rounded-lg shadow-lg">
                {generatedImage ? (
                     <img src={generatedImage} alt="Virtual try-on result" className="w-full h-full object-cover rounded-lg"/>
                ) : (
                    <>
                        <img src={avatar.avatarUrl} alt="User avatar" className="w-full h-full object-cover rounded-lg"/>
                        {/* Garment Layers - rendered in order for natural stacking */}
                        <div className="absolute inset-0 flex items-center justify-center">
                            {/* Base layers. If a dress is chosen, these are not shown. */}
                            {selectedBottom && !selectedDress && <img src={selectedBottom.frontImageUrl} alt={selectedBottom.name} className="absolute w-full h-full object-contain" />}
                            {selectedTop && !selectedDress && <img src={selectedTop.frontImageUrl} alt={selectedTop.name} className="absolute w-full h-full object-contain" />}

                            {/* A dress covers the base layers */}
                            {selectedDress && <img src={selectedDress.frontImageUrl} alt={selectedDress.name} className="absolute w-full h-full object-contain" />}

                            {/* Outerwear is always on top */}
                            {selectedOuterwear && <img src={selectedOuterwear.frontImageUrl} alt={selectedOuterwear.name} className="absolute w-full h-full object-contain" />}
                        </div>
                    </>
                )}
                {isGenerating && (
                    <div className="absolute inset-0 bg-black/60 flex items-center justify-center rounded-lg">
                        <LoadingSpinner text="Creating your look..."/>
                    </div>
                )}
            </div>
            <div className="mt-6 flex flex-col sm:flex-row items-center space-y-4 sm:space-y-0 sm:space-x-4">
                 <button
                    onClick={handleGenerateTryOn}
                    disabled={isGenerating || selectedItemsCount === 0}
                    className="bg-pink-600 text-white font-bold py-3 px-6 rounded-lg hover:bg-pink-700 transition-all shadow-md disabled:bg-stone-400 disabled:cursor-not-allowed w-full sm:w-auto"
                >
                    {isGenerating ? 'Generating...' : 'Show How I Look'}
                </button>
                {generatedImage && (
                     <button
                        onClick={() => setGeneratedImage(null)}
                        className="bg-white text-stone-700 font-bold py-3 px-6 rounded-lg hover:bg-stone-200 transition-all border border-stone-300 w-full sm:w-auto"
                    >
                        Reset
                    </button>
                )}
            </div>
        </div>
    </div>
  );
};

export default VirtualTryOn;
