import React, { useState, useRef, useCallback } from 'react';
import type { Garment } from '../../types';
import { GarmentCategory } from '../../types';
import { analyzeGarmentImage } from '../../services/geminiService';
import LoadingSpinner from '../common/LoadingSpinner';
import { CameraIcon, BackIcon, CheckCircleIcon } from '../common/Icons';

interface AddItemFormProps {
  onAddItem: (item: Garment) => void;
  onBack: () => void;
}

const AddItemForm: React.FC<AddItemFormProps> = ({ onAddItem, onBack }) => {
  const [name, setName] = useState('');
  const [category, setCategory] = useState<GarmentCategory>(GarmentCategory.UNKNOWN);
  const [brand, setBrand] = useState('');
  const [size, setSize] = useState('');
  const [color, setColor] = useState('');
  const [price, setPrice] = useState<number | ''>('');
  const [tags, setTags] = useState<string[]>([]);
  const [tagInput, setTagInput] = useState('');
  const [frontImage, setFrontImage] = useState<File | null>(null);
  const [frontImageUrl, setFrontImageUrl] = useState<string | null>(null);
  const [careInstructions, setCareInstructions] = useState('');
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [isSubmitted, setIsSubmitted] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const handleImageUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      setFrontImage(file);
      const reader = new FileReader();
      reader.onload = (event) => {
        setFrontImageUrl(event.target?.result as string);
      };
      reader.readAsDataURL(file);
      handleImageAnalysis(file);
    }
  };

  const handleImageAnalysis = useCallback(async (file: File) => {
    setIsAnalyzing(true);
    try {
      const result = await analyzeGarmentImage(file);
      if (result) {
        setName(result.name);
        // Map string to enum value, fallback to UNKNOWN
        const foundCategory = Object.values(GarmentCategory).find(c => c.toLowerCase() === result.category.toLowerCase());
        setCategory(foundCategory || GarmentCategory.UNKNOWN);
        setColor(result.color);
        setTags(result.tags);
        setCareInstructions(result.careInstructions);
      }
    } catch (error) {
      console.error('Analysis failed:', error);
      // You could show an error toast here
    } finally {
      setIsAnalyzing(false);
    }
  }, []);

  const handleTagKeyDown = (e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === 'Enter' && tagInput.trim() !== '') {
      e.preventDefault();
      setTags([...tags, tagInput.trim()]);
      setTagInput('');
    }
  };
  
  const removeTag = (indexToRemove: number) => {
    setTags(tags.filter((_, index) => index !== indexToRemove));
  };

  const resetForm = useCallback(() => {
    setName('');
    setCategory(GarmentCategory.UNKNOWN);
    setBrand('');
    setSize('');
    setColor('');
    setPrice('');
    setTags([]);
    setTagInput('');
    setFrontImage(null);
    setFrontImageUrl(null);
    setCareInstructions('');
    if (fileInputRef.current) {
        fileInputRef.current.value = '';
    }
  }, []);

  const createNewItem = (): Garment | null => {
    if (!frontImageUrl || !name) {
        alert("Please upload an image and provide a name for the garment.");
        return null;
    }
    return {
      id: Date.now().toString(),
      name,
      category,
      brand,
      size,
      color,
      price: Number(price) || 0,
      purchaseDate: new Date().toISOString(),
      tags,
      frontImageUrl,
      careInstructions
    };
  };

  const handleSaveAndAddAnother = () => {
    const newItem = createNewItem();
    if (newItem) {
        onAddItem(newItem);
        resetForm();
        setIsSubmitted(true);
        setTimeout(() => setIsSubmitted(false), 3000);
    }
  };

  const handleSaveAndFinish = () => {
      const newItem = createNewItem();
      if(newItem) {
          onAddItem(newItem);
          onBack();
      }
  };

  return (
    <div className="p-4 max-w-2xl mx-auto pb-24">
      {isSubmitted && (
        <div className="fixed top-16 left-1/2 -translate-x-1/2 w-11/12 max-w-sm bg-green-500 text-white px-6 py-4 rounded-xl shadow-lg z-50 flex items-center space-x-4" role="alert">
            <CheckCircleIcon className="w-8 h-8"/>
            <div>
                <p className="font-bold">Success!</p>
                <p className="text-sm">Item added to your wardrobe.</p>
            </div>
        </div>
      )}
      <div className="flex items-center mb-6">
        <button onClick={onBack} className="p-2 rounded-full hover:bg-stone-200 mr-2">
            <BackIcon className="h-6 w-6"/>
        </button>
        <h1 className="text-3xl font-bold">Add New Purchase</h1>
      </div>

      <form onSubmit={(e) => e.preventDefault()} className="space-y-6">
        <div 
          className="relative w-full h-64 border-2 border-dashed border-stone-300 rounded-lg flex items-center justify-center cursor-pointer bg-stone-100 hover:bg-stone-200 transition-colors"
          onClick={() => fileInputRef.current?.click()}
        >
          {frontImageUrl ? (
            <img src={frontImageUrl} alt="Garment front" className="h-full w-full object-contain rounded-lg" />
          ) : (
            <div className="text-center text-stone-500">
              <CameraIcon className="mx-auto h-12 w-12" />
              <p>Upload a photo of your item</p>
              <p className="text-sm">We'll try to fill in the details!</p>
            </div>
          )}
          {isAnalyzing && (
            <div className="absolute inset-0 bg-white/80 flex items-center justify-center rounded-lg">
                <LoadingSpinner text="Analyzing..."/>
            </div>
          )}
        </div>
        <input type="file" accept="image/*" capture="environment" ref={fileInputRef} onChange={handleImageUpload} className="hidden" />

        {/* Form Fields */}
        <div>
          <label htmlFor="name" className="block text-sm font-medium text-stone-700">Item Name</label>
          <input type="text" id="name" value={name} onChange={(e) => setName(e.target.value)} required className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"/>
        </div>
        
        <div>
          <label htmlFor="category" className="block text-sm font-medium text-stone-700">Category</label>
          <select id="category" value={category} onChange={(e) => setCategory(e.target.value as GarmentCategory)} className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500">
            {Object.values(GarmentCategory).map(cat => <option key={cat} value={cat}>{cat}</option>)}
          </select>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label htmlFor="brand" className="block text-sm font-medium text-stone-700">Brand</label>
              <input type="text" id="brand" value={brand} onChange={(e) => setBrand(e.target.value)} className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"/>
            </div>
             <div>
              <label htmlFor="size" className="block text-sm font-medium text-stone-700">Size</label>
              <input type="text" id="size" value={size} onChange={(e) => setSize(e.target.value)} className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"/>
            </div>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
             <div>
              <label htmlFor="color" className="block text-sm font-medium text-stone-700">Color</label>
              <input type="text" id="color" value={color} onChange={(e) => setColor(e.target.value)} className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"/>
            </div>
            <div>
              <label htmlFor="price" className="block text-sm font-medium text-stone-700">Price (₹)</label>
              <input type="number" id="price" value={price} onChange={(e) => setPrice(e.target.value === '' ? '' : Number(e.target.value))} className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"/>
            </div>
        </div>

        <div>
            <label htmlFor="tags" className="block text-sm font-medium text-stone-700">Tags</label>
            <div className="mt-1 flex flex-wrap gap-2 items-center p-2 border border-stone-300 rounded-md">
                {tags.map((tag, index) => (
                    <span key={index} className="bg-pink-100 text-pink-800 text-sm font-medium px-2.5 py-0.5 rounded-full flex items-center">
                        {tag}
                        <button type="button" onClick={() => removeTag(index)} className="ml-1.5 text-pink-500 hover:text-pink-700">
                            &times;
                        </button>
                    </span>
                ))}
                <input
                    type="text"
                    id="tags"
                    value={tagInput}
                    onChange={(e) => setTagInput(e.target.value)}
                    onKeyDown={handleTagKeyDown}
                    placeholder="Add tags..."
                    className="flex-grow bg-transparent focus:outline-none"
                />
            </div>
        </div>
        
        <div>
          <label htmlFor="care" className="block text-sm font-medium text-stone-700">Care Instructions</label>
          <textarea id="care" value={careInstructions} onChange={(e) => setCareInstructions(e.target.value)} rows={3} className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"></textarea>
        </div>

        <div className="flex flex-col sm:flex-row-reverse space-y-4 space-y-reverse sm:space-y-0 sm:space-x-4 sm:space-x-reverse pt-2">
            <button 
                type="button" 
                onClick={handleSaveAndFinish}
                className="w-full bg-pink-600 text-white font-bold py-3 px-4 rounded-lg hover:bg-pink-700 transition-all shadow-md"
            >
                Save & Finish
            </button>
            <button 
                type="button" 
                onClick={handleSaveAndAddAnother}
                className="w-full bg-white text-pink-600 font-bold py-3 px-4 rounded-lg hover:bg-pink-50 transition-all border border-pink-600"
            >
                Save & Add Another
            </button>
        </div>
      </form>
    </div>
  );
};

export default AddItemForm;