import React, { useState } from 'react';
import type { Garment, WearEvent, AppView } from '../../types';
import { BackIcon, CheckCircleIcon, PlusIcon, WardrobeIcon } from '../common/Icons';

interface LogWearEventFormProps {
  wardrobe: Garment[];
  onLogEvent: (event: Omit<WearEvent, 'id'>) => void;
  onBack: () => void;
  setView: (view: AppView) => void;
}

const LogWearEventForm: React.FC<LogWearEventFormProps> = ({ wardrobe, onLogEvent, onBack, setView }) => {
  const [selectedGarmentIds, setSelectedGarmentIds] = useState<string[]>([]);
  const [date, setDate] = useState(new Date().toISOString().split('T')[0]); // YYYY-MM-DD
  const [occasion, setOccasion] = useState('');
  const [notes, setNotes] = useState('');

  const toggleGarmentSelection = (garmentId: string) => {
    setSelectedGarmentIds(prev =>
      prev.includes(garmentId)
        ? prev.filter(id => id !== garmentId)
        : [...prev, garmentId]
    );
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (selectedGarmentIds.length === 0) {
      alert('Please select at least one garment.');
      return;
    }
    onLogEvent({
      garmentIds: selectedGarmentIds,
      date,
      occasion,
      notes,
    });
  };

  return (
    <div className="p-4 max-w-2xl mx-auto pb-24">
      <div className="flex items-center mb-6">
        <button onClick={onBack} className="p-2 rounded-full hover:bg-stone-200 mr-2">
          <BackIcon className="h-6 w-6" />
        </button>
        <h1 className="text-3xl font-bold">Log Outfit</h1>
      </div>

      <form onSubmit={handleSubmit} className="space-y-6">
        <div>
          <div className="flex justify-between items-center mb-2">
            <label className="block text-sm font-medium text-stone-700">Select Garments</label>
             <button
                type="button"
                onClick={() => setView('ADD_ITEM')}
                className="text-sm text-pink-600 font-semibold hover:text-pink-800 flex items-center gap-1"
            >
                <PlusIcon className="w-4 h-4" />
                Add New Item
            </button>
          </div>
          {wardrobe.length > 0 ? (
            <div className="grid grid-cols-3 sm:grid-cols-4 md:grid-cols-5 gap-2">
              {wardrobe.map(garment => (
                <div
                  key={garment.id}
                  onClick={() => toggleGarmentSelection(garment.id)}
                  className="relative cursor-pointer rounded-lg overflow-hidden border-2 border-transparent hover:border-pink-400"
                >
                  <img src={garment.frontImageUrl} alt={garment.name} className="w-full h-24 object-cover" />
                  {selectedGarmentIds.includes(garment.id) && (
                    <div className="absolute inset-0 bg-black/50 flex items-center justify-center">
                      <CheckCircleIcon className="h-8 w-8 text-white" />
                    </div>
                  )}
                </div>
              ))}
            </div>
          ) : (
            <div className="text-center py-10 flex flex-col items-center justify-center bg-stone-100 rounded-lg">
                <WardrobeIcon className="w-20 h-20 text-stone-300 mb-4" />
                <h2 className="text-xl font-semibold text-stone-700 mb-2">Your wardrobe is empty!</h2>
                <p className="text-stone-500 mb-6 max-w-xs mx-auto">Add some clothes to your virtual closet before you can log an outfit.</p>
                <button
                    type="button"
                    onClick={() => setView('ADD_ITEM')}
                    className="bg-pink-600 text-white font-bold py-2 px-5 rounded-lg hover:bg-pink-700 transition-all flex items-center gap-2 shadow-md"
                >
                    <PlusIcon className="w-5 h-5" />
                    Add First Item
                </button>
            </div>
          )}
        </div>

        {wardrobe.length > 0 && (
            <>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label htmlFor="date" className="block text-sm font-medium text-stone-700">Date</label>
                    <input
                    type="date"
                    id="date"
                    value={date}
                    onChange={e => setDate(e.target.value)}
                    required
                    className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"
                    />
                </div>
                <div>
                    <label htmlFor="occasion" className="block text-sm font-medium text-stone-700">Occasion</label>
                    <input
                    type="text"
                    id="occasion"
                    value={occasion}
                    onChange={e => setOccasion(e.target.value)}
                    placeholder="e.g., Work, Casual, Party"
                    required
                    className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"
                    />
                </div>
                </div>
                
                <div>
                <label htmlFor="notes" className="block text-sm font-medium text-stone-700">Notes</label>
                <textarea
                    id="notes"
                    value={notes}
                    onChange={e => setNotes(e.target.value)}
                    rows={3}
                    placeholder="e.g., Coffee with friends"
                    className="mt-1 block w-full px-3 py-2 bg-white border border-stone-300 rounded-md shadow-sm focus:outline-none focus:ring-pink-500 focus:border-pink-500"
                ></textarea>
                </div>

                <button
                type="submit"
                className="w-full bg-pink-600 text-white font-bold py-3 px-4 rounded-lg hover:bg-pink-700 transition-all shadow-md disabled:bg-stone-400"
                disabled={selectedGarmentIds.length === 0}
                >
                Log Outfit
                </button>
            </>
        )}
      </form>
    </div>
  );
};

export default LogWearEventForm;