import { GoogleGenAI, Type, Modality } from "@google/genai";
import type { GenerateContentResponse } from "@google/genai";

// Ensure API_KEY is available. In a real app, this would be handled more securely.
const API_KEY = process.env.API_KEY;
if (!API_KEY) {
  console.warn("API_KEY not found in environment variables. Gemini features will be disabled.");
}

const ai = new GoogleGenAI({ apiKey: API_KEY! });

// Utility to convert file to base64
const fileToGenerativePart = async (file: File) => {
  const base64EncodedDataPromise = new Promise<string>((resolve) => {
    const reader = new FileReader();
    reader.onloadend = () => resolve((reader.result as string).split(',')[1]);
    reader.readAsDataURL(file);
  });
  return {
    inlineData: { data: await base64EncodedDataPromise, mimeType: file.type },
  };
};

// Helper to fetch an image from a URL and convert it to base64
const imageUrlToBase64 = async (url: string): Promise<{data: string, mimeType: string}> => {
  if (url.startsWith('data:')) {
    const [meta, data] = url.split(',');
    if (!meta || !data) {
      throw new Error("Invalid data URL format");
    }
    const mimeType = meta.split(':')[1].split(';')[0];
    return { data, mimeType };
  }
  
  const response = await fetch(url);
  if (!response.ok) {
    throw new Error(`HTTP error! status: ${response.status}`);
  }
  const blob = await response.blob();
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onloadend = () => {
      const result = reader.result as string;
      const [, data] = result.split(',');
      if (!data) {
        return reject(new Error("Invalid data URL format from fetched blob"));
      }
      resolve({ data, mimeType: blob.type });
    };
    reader.onerror = reject;
    reader.readAsDataURL(blob);
  });
};


interface GarmentAnalysis {
    category: string;
    color: string;
    tags: string[];
    name: string;
    careInstructions: string;
}

export const analyzeGarmentImage = async (imageFile: File): Promise<GarmentAnalysis | null> => {
  if (!API_KEY) return null;
  try {
    const imagePart = await fileToGenerativePart(imageFile);
    const response: GenerateContentResponse = await ai.models.generateContent({
      model: 'gemini-2.5-flash',
      contents: { 
        parts: [
            imagePart,
            { text: "Analyze this image of a garment. Identify its category (e.g., 'Top', 'Jeans', 'Dress'), primary color, and suggest 3 relevant tags (e.g., 'casual', 'summer', 'cotton'). Also, provide a concise descriptive name (e.g., 'Blue Floral Sundress') and suggest common care instructions. Return a single, valid JSON object." }
        ] 
      },
      config: {
        responseMimeType: "application/json",
        responseSchema: {
            type: Type.OBJECT,
            properties: {
                category: { type: Type.STRING, description: "Category of the garment (Top, Bottom, Dress, Outerwear, Accessory, Shoes)." },
                color: { type: Type.STRING, description: "Primary color of the garment." },
                tags: { type: Type.ARRAY, items: { type: Type.STRING }, description: "List of relevant tags." },
                name: { type: Type.STRING, description: "A concise, descriptive name for the garment." },
                careInstructions: { type: Type.STRING, description: "Common care instructions, e.g., 'Machine wash cold, tumble dry low'."}
            }
        }
      }
    });
    
    const text = response.text.trim();
    return JSON.parse(text) as GarmentAnalysis;
  } catch (error) {
    console.error("Error analyzing garment image:", error);
    return null;
  }
};

interface NextPurchaseSuggestion {
    suggestion: string;
    reason: string;
}

export const suggestNextPurchase = async (wardrobe: string[]): Promise<NextPurchaseSuggestion | null> => {
    if (!API_KEY) return null;
    try {
        const prompt = `You are a fashion stylist. Analyze this detailed wardrobe list: ${wardrobe.join('; ')}. 
        Identify a single, specific, versatile clothing item that is missing and would significantly increase the number of possible outfits. 
        Be creative and avoid generic suggestions like "white t-shirt", "black pants", or "sneakers" unless the wardrobe is truly lacking absolute basics.
        Instead of "a blazer", suggest "a camel-colored oversized wool blazer". 
        Your suggestion should complement the existing items' styles and colors.
        Explain your reasoning in one concise sentence. Return a single, valid JSON object.`;

        const response: GenerateContentResponse = await ai.models.generateContent({
            model: 'gemini-2.5-flash',
            contents: prompt,
            config: {
                temperature: 0.9,
                responseMimeType: "application/json",
                responseSchema: {
                    type: Type.OBJECT,
                    properties: {
                        suggestion: { type: Type.STRING, description: "The specific suggested item, e.g., 'A camel-colored oversized wool blazer'." },
                        reason: { type: Type.STRING, description: "A brief reason for the suggestion." }
                    }
                }
            }
        });
        
        const text = response.text.trim();
        return JSON.parse(text) as NextPurchaseSuggestion;
    } catch (error) {
        console.error("Error suggesting next purchase:", error);
        return null;
    }
};

export interface OutfitSuggestion {
    outfitItems: string[];
    reason: string;
}

export const suggestOutfitFromWardrobe = async (wardrobe: string[]): Promise<OutfitSuggestion | null> => {
    if (!API_KEY) return null;
    try {
        const prompt = `You are a fashion stylist. Analyze this wardrobe: ${wardrobe.join('; ')}. 
        Create a single, stylish, and complete outfit combination using ONLY items from the provided list.
        The outfit should ideally consist of a top, a bottom (or a dress), and maybe outerwear or shoes if available.
        Explain your reasoning for this combination in one concise sentence. Return a single, valid JSON object.`;

        const response: GenerateContentResponse = await ai.models.generateContent({
            model: 'gemini-2.5-flash',
            contents: prompt,
            config: {
                temperature: 0.7,
                responseMimeType: "application/json",
                responseSchema: {
                    type: Type.OBJECT,
                    properties: {
                        outfitItems: { 
                            type: Type.ARRAY, 
                            items: { type: Type.STRING },
                            description: "An array of garment names from the wardrobe that form the outfit." 
                        },
                        reason: { type: Type.STRING, description: "A brief reason why this outfit works well." }
                    }
                }
            }
        });
        
        const text = response.text.trim();
        return JSON.parse(text) as OutfitSuggestion;
    } catch (error) {
        console.error("Error suggesting outfit from wardrobe:", error);
        return null;
    }
};

export const generateVirtualTryOnImage = async (
    userPhotoBase64WithPrefix: string,
    garments: { name: string; imageUrl: string }[]
): Promise<string | null> => {
    if (!API_KEY) return null;
    try {
        const [userPhotoMeta, userPhotoData] = userPhotoBase64WithPrefix.split(',');
        if (!userPhotoMeta || !userPhotoData) throw new Error("Invalid user photo format");
        const userPhotoMimeType = userPhotoMeta.split(':')[1].split(';')[0];

        const userImagePart = {
            inlineData: {
                data: userPhotoData,
                mimeType: userPhotoMimeType,
            }
        };

        const garmentImageParts = await Promise.all(
            garments.map(async (garment) => {
                const { data, mimeType } = await imageUrlToBase64(garment.imageUrl);
                return { inlineData: { data, mimeType } };
            })
        );
        
        const garmentNames = garments.map(g => g.name).join(', ');
        const textPart = {
            text: `This is for a virtual try-on. Realistically and seamlessly place the clothing item(s) (${garmentNames}) from the following images onto the person in the main photo. The main photo is the one with the person. The other images are the clothes. Make sure the clothes fit the person's body naturally. Pay close attention to fit, drape, and lighting to make it look as natural as possible.`
        };

        const response: GenerateContentResponse = await ai.models.generateContent({
            model: 'gemini-2.5-flash-image-preview',
            contents: {
                parts: [
                    userImagePart,
                    ...garmentImageParts,
                    textPart
                ],
            },
            config: {
                responseModalities: [Modality.IMAGE, Modality.TEXT],
            },
        });

        for (const part of response.candidates[0].content.parts) {
            if (part.inlineData) {
                const base64ImageBytes: string = part.inlineData.data;
                const mimeType = part.inlineData.mimeType;
                return `data:${mimeType};base64,${base64ImageBytes}`;
            }
        }
        
        console.warn("No image found in Gemini response for virtual try-on.");
        return null;

    } catch (error) {
        console.error("Error generating virtual try-on image:", error);
        return null;
    }
};

export interface TravelOutfitPlan {
    day: number;
    occasion: string;
    outfitItems: string[];
}

export const planTravelOutfits = async (
    destination: string,
    durationInDays: number,
    travelType: string,
    wardrobe: string[]
): Promise<TravelOutfitPlan[] | null> => {
    if (!API_KEY) return null;
    try {
        const prompt = `You are a travel stylist. Based on the following travel details and a wardrobe, create a day-by-day outfit plan.
        Use ONLY items from the provided wardrobe list.
        - Destination: ${destination}
        - Duration: ${durationInDays} days
        - Trip Type: ${travelType}
        - Wardrobe: ${wardrobe.join(', ')}

        For each day, suggest an outfit for a likely activity (e.g., 'Sightseeing', 'Dinner').
        Return a valid JSON array where each object represents a day's plan.
        `;

        const response: GenerateContentResponse = await ai.models.generateContent({
            model: 'gemini-2.5-flash',
            contents: prompt,
            config: {
                responseMimeType: "application/json",
                responseSchema: {
                    type: Type.ARRAY,
                    items: {
                        type: Type.OBJECT,
                        properties: {
                            day: { type: Type.NUMBER, description: "The day number of the trip (e.g., 1, 2)." },
                            occasion: { type: Type.STRING, description: "A suggested occasion or activity for the day, e.g., 'Sightseeing', 'Evening Dinner'." },
                            outfitItems: {
                                type: Type.ARRAY,
                                items: { type: Type.STRING },
                                description: "A list of garment names from the provided wardrobe for the outfit."
                            }
                        }
                    }
                }
            }
        });

        const text = response.text.trim();
        return JSON.parse(text) as TravelOutfitPlan[];

    } catch (error) {
        console.error("Error planning travel outfits:", error);
        return null;
    }
};