
import React, { useState, useCallback } from 'react';
import { AppState } from './types';
import { extractTextFromImage } from './services/geminiService';
import { toBase64 } from './utils/imageUtils';
import ImageUploader from './components/ImageUploader';
import Spinner from './components/Spinner';
import ExtractedTextView from './components/ExtractedTextView';

const App: React.FC = () => {
  const [appState, setAppState] = useState<AppState>(AppState.IDLE);
  const [imagePreviewUrl, setImagePreviewUrl] = useState<string | null>(null);
  const [extractedText, setExtractedText] = useState<string>('');
  const [error, setError] = useState<string>('');

  const resetState = () => {
    setExtractedText('');
    setError('');
    setImagePreviewUrl(null);
    setAppState(AppState.IDLE);
  };
  
  const handleFileSelect = useCallback(async (file: File) => {
    const validTypes = ['image/jpeg', 'image/png', 'image/webp'];
    if (!validTypes.includes(file.type)) {
      setError('Invalid file type. Please upload a PNG, JPG, or WEBP image.');
      setAppState(AppState.ERROR);
      return;
    }

    setAppState(AppState.LOADING);
    setError('');
    
    const objectURL = URL.createObjectURL(file);
    setImagePreviewUrl(objectURL);

    try {
      const base64Image = await toBase64(file);
      const text = await extractTextFromImage(base64Image, file.type);
      setExtractedText(text);
      setAppState(AppState.SUCCESS);
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'An unknown error occurred.';
      setError(`Failed to process image. ${errorMessage}`);
      setAppState(AppState.ERROR);
    }
  }, []);


  return (
    <div className="bg-gray-100 min-h-screen w-full flex flex-col items-center p-4 sm:p-6 lg:p-8">
      <div className="w-full max-w-6xl mx-auto">
        <header className="text-center mb-8">
          <h1 className="text-4xl sm:text-5xl font-bold text-gray-800">Image Text Extractor</h1>
          <p className="mt-2 text-lg text-gray-600">Upload an image to automatically extract text with AI.</p>
        </header>

        <main className="bg-white rounded-2xl shadow-lg p-6 sm:p-8">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 items-start">
            
            {/* Left Column: Uploader / Image Preview */}
            <div className="w-full">
              <h2 className="text-xl font-semibold text-gray-800 mb-2">
                {imagePreviewUrl ? "Your Image" : "Upload Image"}
              </h2>
              {appState === AppState.IDLE || appState === AppState.ERROR ? (
                <ImageUploader onFileSelect={handleFileSelect} disabled={false} />
              ) : (
                 <div className="relative w-full aspect-video rounded-lg overflow-hidden border border-gray-200">
                    {imagePreviewUrl && <img src={imagePreviewUrl} alt="Preview" className="w-full h-full object-contain" />}
                    {appState === AppState.LOADING && (
                      <div className="absolute inset-0 bg-black bg-opacity-50 flex flex-col items-center justify-center">
                        <Spinner />
                        <p className="text-white mt-2 font-semibold">Extracting text...</p>
                      </div>
                    )}
                 </div>
              )}
               {error && (
                <div className="mt-4 p-3 bg-red-100 border border-red-400 text-red-700 rounded-lg">
                  <p className="font-bold">Error</p>
                  <p>{error}</p>
                </div>
              )}
            </div>

            {/* Right Column: Extracted Text */}
            <div className="w-full">
              {appState === AppState.SUCCESS && (
                <ExtractedTextView text={extractedText} />
              )}
            </div>

          </div>

           {(appState === AppState.SUCCESS || appState === AppState.ERROR) && (
             <div className="mt-8 text-center">
                <button 
                  onClick={resetState}
                  className="px-6 py-2 bg-indigo-600 text-white font-semibold rounded-lg shadow-md hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 transition-all"
                >
                  Start Over
                </button>
             </div>
           )}

        </main>
        
        <footer className="text-center mt-8 text-gray-500">
            <p>Powered by Google Gemini</p>
        </footer>
      </div>
    </div>
  );
};

export default App;
