
import React, { useState, useEffect } from 'react';
import { CopyIcon, CheckIcon } from './icons';

interface ExtractedTextViewProps {
  text: string;
}

const ExtractedTextView: React.FC<ExtractedTextViewProps> = ({ text }) => {
  const [isCopied, setIsCopied] = useState(false);

  useEffect(() => {
    if (isCopied) {
      const timer = setTimeout(() => {
        setIsCopied(false);
      }, 2000);
      return () => clearTimeout(timer);
    }
  }, [isCopied]);

  const handleCopy = () => {
    navigator.clipboard.writeText(text);
    setIsCopied(true);
  };

  return (
    <div className="relative w-full">
        <h2 className="text-xl font-semibold text-gray-800 mb-2">Extracted Text</h2>
        <div className="w-full bg-gray-50 rounded-lg p-4 border border-gray-200 min-h-[16rem] max-h-96 overflow-y-auto">
            <pre className="text-gray-700 whitespace-pre-wrap font-sans">{text || "No text extracted."}</pre>
        </div>
        <button
            onClick={handleCopy}
            className="absolute top-0 right-0 mt-2 mr-2 p-2 rounded-lg bg-gray-200 hover:bg-gray-300 transition-colors focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
            aria-label="Copy text"
        >
            {isCopied ? (
                <CheckIcon className="w-5 h-5 text-green-600" />
            ) : (
                <CopyIcon className="w-5 h-5 text-gray-600" />
            )}
        </button>
    </div>
  );
};

export default ExtractedTextView;
